<?php

namespace Drupal\ai_provider_voyage\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiSetupFailureException;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsOutput;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Exception;

/**
 * Plugin implementation of the 'voyage' provider.
 */
#[AiProvider(
  id: 'voyage',
  label: new TranslatableMarkup('Voyage AI')
)]
class VoyageProvider extends OpenAiBasedProviderClientBase {

//  use ChatTrait;

  /**
   * {@inheritdoc}
   */
  protected string $endpoint = 'https://api.voyageai.com/v1';

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {

    $models = $this->getHardcodedModels($operation_type, $capabilities);

    return $models;
  }

  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
      return in_array($operation_type, $this->getSupportedOperationTypes());
  }
  
  public function embeddings(string|EmbeddingsInput $input, string $model_id, array $tags = []): EmbeddingsOutput {
    $this->loadClient();

    if ($input instanceof EmbeddingsInput) {
      $input = $input->getPrompt();
    }
    
    $configuration = $this->configuration;
    if (isset($configuration['input_type']) && empty($configuration['input_type'])) {
      unset($configuration['input_type']);
    }        

    $payload = [
      'model' => $model_id,
      'input' => $input,
    ] + $configuration;

    try {
      $response = $this->client->embeddings()->create($payload)->toArray();
      return new EmbeddingsOutput($response['data'][0]['embedding'], $response, []);
    }
    catch (Exception $e) {
      $this->handleApiException($e);
      throw $e;
    }
  }
  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'embeddings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }
  
  /**
   * {@inheritdoc}
   */
  protected function loadClient(): void {
    // Set custom endpoint from host config if available.
    if (!empty($this->getConfig()->get('host'))) {
      $this->setEndpoint($this->getConfig()->get('host'));
    }

    try {
      parent::loadClient();
    }
    catch (AiSetupFailureException $e) {
      throw new AiSetupFailureException('Failed to initialize Voyage client: ' . $e->getMessage(), $e->getCode(), $e);
    }
  }

  /**
   * Returns hardcoded models for backward compatibility.
   *
   * @param string|null $operation_type
   *   The operation type.
   * @param array $capabilities
   *   Required capabilities.
   *
   * @return array
   *   Array of hardcoded models.
   */
  protected function getHardcodedModels(?string $operation_type = NULL, array $capabilities = []): array {

    // These are the existing hardcoded models.
    $models = [
      'voyage-3-large' => 'Voyage 3 Large',
      'voyage-3.5' => 'Voyage 3.5',
      'voyage-3.5-lite' => 'Voyage 3.5 Lite',
      'voyage-code-3' => 'Voyage Code 3',
      'voyage-finance-2' => 'Voyage Finance 2',
      'voyage-law-2' => 'Voyage Law 2',
    ];

    return $models;
  }
  
  public function getSetupData(): array {
    return [
      'key_config_name' => 'api_key',
      'default_models' => [
        'embeddings' => 'voyage-3.5',
      ],
    ];
  }


  /**
   * Handle API exceptions consistently.
   *
   * @param Exception $e
   *   The exception to handle.
   *
   * @throws AiRateLimitException
   * @throws AiQuotaException
   * @throws Exception
   */
  protected function handleApiException(Exception $e): void {
    if (strpos($e->getMessage(), 'Rate Limit Exceeded') !== FALSE) {
      throw new AiRateLimitException($e->getMessage());
    }
    throw $e;
  }
  
   /**
   * {@inheritdoc}
   */
  public function loadModelsForm(array $form, $form_state, string $operation_type, string|NULL $model_id = NULL): array {
    // maybe extend later
    return parent::loadModelsForm($form, $form_state, $operation_type, $model_id);
  }

}
