<?php

namespace Drupal\ai_provider_yandex\Plugin\AiProvider;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;

/**
 * Plugin implementation of the 'yandex' provider.
 */
#[AiProvider(
  id: 'yandex',
  label: new TranslatableMarkup('Yandex'),
)]
class YandexProvider extends OpenAiBasedProviderClientBase {

  use ChatTrait;

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    return [
      'yandexgpt-lite/latest' => 'YandexGPT Lite (Latest 5)',
      'yandexgpt-lite/rc' => 'YandexGPT Lite (RC 5)',
      'yandexgpt/latest' => 'YandexGPT Pro (Latest 5)',
      'yandexgpt/rc' => 'YandexGPT Pro (RC 5.1)',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // If it is not configured, it is not usable.
    if (!$this->apiKey && !$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    // If its one of the bundles that YandexGPT supports its usable.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $this->setEndpoint('https://llm.api.cloud.yandex.net/v1');
    $catalog_id = $this->getConfig()->get('catalog_id');
    $model_id = "gpt://$catalog_id/$model_id";
    return parent::chat($input, $model_id, $tags);
  }

}
