<?php

namespace Drupal\ai_provider_zhipuai\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai_provider_zhipuai\ZhipuaiClient;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the Zhipuai LLM.
 */
#[AiProvider(
  id: 'zhipuai',
  label: new TranslatableMarkup('Zhipuai')
)]
class ZhipuaiProvider extends AiProviderClientBase implements ChatInterface
{

  /**
   * Zhipuai client
   *
   * @var ZhipuaiClient|null
   */
  protected ZhipuaiClient|null $client = null;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * Run moderation call, before a normal call.
   *
   * @var bool
   */
  protected bool $moderation = TRUE;

  /**
   * {@inheritdoc}
   * @param string|null $operation_type
   * @param array $capabilities
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool
  {
    if (!$this->getConfig()->get('api_key')) {
      return FALSE;
    }

    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array
  {
    return ['chat'];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig
  {
    return $this->configFactory->get('ai_provider_zhipuai.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array
  {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void
  {
    $this->apiKey = $authentication;
  }

  /**
   * Gets the raw client.
   *
   * @param string $api_key
   *   If the API key should be hot swapped.
   *
   * @return ZhipuaiClient
   *   The Zhipuai client.
   */
  public function getClient(string $api_key = '')
  {
    if ($api_key) {
      $this->setAuthentication($api_key);
    }

    $this->loadClient();
    return $this->client;
  }

  /**
   * Loads the Zhipuai Client with authentication if not initialized.
   */
  protected function loadClient(): void
  {
    if (!$this->client) {
      if (!$this->apiKey) {
        $this->setAuthentication($this->loadApiKey());
      }
      $this->client = \Drupal::service('ai_provider_zhipuai.client');
      $this->client->setApiToken($this->apiKey);
    }
  }

  /**
   * {@inheritdoc}
   * @param string|null $operation_type
   * @param array $capabilities
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array
  {
    switch ($operation_type) {
      case 'chat':
        $models = [
          "glm-4.5",
          "glm-4.5-air",
          "glm-4.5-x",
          "glm-4.5-airx",
          "glm-4.5-flash",
        ];
        break;
      default:
        $models = [];
    }
    return array_combine($models, $models);
  }

  /**
   * Load API key from key module.
   *
   * @return string
   *   The API key.
   */
  protected function loadApiKey(): string
  {
    return $this->keyRepository->getKey($this->getConfig()->get('api_key'))
      ->getKeyValue();
  }

  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput
  {
    $this->loadClient();

    $response = $this->client->textGeneration('/chat/completions', $model_id, $input);
    $data = Json::decode($response);
    $message = new ChatMessage($data['choices'][0]['message']['role'] ?? '', $data['choices'][0]['message']['content'] ?? '');

    return new ChatOutput($message, $response, []);
  }

}
