<?php

namespace Drupal\ai_provider_zhipuai;

use Drupal\ai\OperationType\Chat\ChatMessage;
use GuzzleHttp\Client;

/**
 * Basic Zhipuai API.
 */
class ZhipuaiClient
{

  /**
   * The http client.
   */
  protected Client $client;

  /**
   * API Token.
   */
  private string $apiToken;

  /**
   * The serverless base path.
   */
  private string $serverless = 'https://open.bigmodel.cn/api/paas/v4';

  /**
   * Constructs a new Zhipuai object.
   *
   * @param \GuzzleHttp\Client $client
   *   Http client.
   */
  public function __construct(Client $client)
  {
    $this->client = $client;
  }

  /**
   * Set the API token.
   *
   * @param string $apiToken
   *   The API token.
   */
  public function setApiToken($apiToken)
  {
    $this->apiToken = $apiToken;
  }

  /**
   * Checks if the api is set.
   *
   * @return bool
   *   If the api is set.
   */
  public function isApiSet()
  {
    return !empty($this->apiToken);
  }

  /**
   * Makes an Text Generation task call.
   *
   * @param string $endpoint
   * @param string $model_id
   * @param string $prompt
   *
   * @return string
   *   The return response undecoded.
   */
  public function textGeneration($endpoint, $model_id, $prompt)
  {
    $input = $prompt->getMessages()[0]->toArray();
    $apiEndPoint = $this->finalEndpoint($endpoint);
    return $this->makeRequest($apiEndPoint, [
      'model' => $model_id,
      'messages' => [
        [
          'role' => $input['role'],
          'content' => $input['text'],
        ],
      ],
    ]);
  }

  /**
   * Is endpoint a serverless endpoint or a dedicated url.
   *
   * @param string $endpoint
   *   The endpoint url or model name.
   *
   * @return string
   *   The final endpoint.
   */
  protected function finalEndpoint($endpoint)
  {
    // If it has a protocol, it's a dedicated url.
    if (strpos($endpoint, 'https://') === 0 || strpos($endpoint, 'http://') === 0) {
      return $endpoint;
    }
    // Otherwise, it's a serverless endpoint.
    return $this->serverless . $endpoint;
  }

  /**
   * Make Zhipuai call.
   *
   * @param string $apiEndPoint
   *   The api endpoint.
   * @param string $json
   *   JSON params.
   * @param string $file
   *   A (real) filepath.
   * @param string $method
   *   The http method.
   *
   * @return string|object
   *   The return response.
   */
  protected function makeRequest($apiEndPoint, $json = NULL, $file = NULL, $method = 'POST')
  {
    if (empty($this->apiToken)) {
      throw new \Exception('No Zhipuai API token found.');
    }

    // We can wait some.
    $options['connect_timeout'] = 120;
    $options['read_timeout'] = 120;
    // Set authorization header.
    $options['headers']['Authorization'] = 'Bearer ' . $this->apiToken;

    if ($json) {
      $options['body'] = json_encode($json);
      $options['headers']['Content-Type'] = 'application/json';
    }

    if ($file) {
      $options['body'] = fopen($file, 'r');
    }

    $res = $this->client->request($method, $apiEndPoint, $options);
    return $res->getBody();
  }

}
