<?php

namespace Drupal\ai_related_content\Form;

use Drupal\ai_related_content\AiRelatedContentManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Url;
use Drupal\ai_search\Backend\AiSearchBackendPluginBase;
use Drupal\Component\Serialization\Yaml;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for creating the AI Related Content view configuration.
 *
 * Allows selecting a compatible Search API index and generates the view
 * configuration from a template. Handles existing view configuration.
 */
class CreateAiRelatedContent extends FormBase {

  /**
   * Constructs a new the form.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extensionPathResolver
   *   The extension path resolver.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The AI Related Content logger.
   * @param \Drupal\ai_related_content\AiRelatedContentManager $aiRelatedContentManager
   *   The AI Related Content helper.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ExtensionPathResolver $extensionPathResolver,
    protected LoggerChannelInterface $logger,
    protected AiRelatedContentManager $aiRelatedContentManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('extension.path.resolver'),
      $container->get('logger.factory')->get('ai_related_content'),
      $container->get('ai_related_content.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_related_content_create_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $compatible_indexes = $this->aiRelatedContentManager->findCompatibleIndexes();
    if (empty($compatible_indexes)) {
      $form['no_indexes_message'] = [
        '#markup' => $this->t('No compatible Search API indexes found. Please ensure you have a Search API index using a backend that supports AI Search (extends %backend_class).', ['%backend_class' => AiSearchBackendPluginBase::class]),
      ];
      return $form;
    }

    // View details.
    $view_id = 'ai_related_content';
    $view_config_name = 'views.view.' . $view_id;
    $view_exists = !$this->configFactory()->get($view_config_name)->isNew();

    if ($view_exists) {
      $url = Url::fromRoute('entity.view.edit_form', ['view' => $view_id]);
      $form['description'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'status' => [
            $this->t('The <a href="@edit_link">AI Related Content View</a> exists. you can add the View as a Block where desired. Alternatively, you can use the Twig Tweak module to render the block with <code>{{ drupal_view(\'ai_related_content\', \'default_block\') }}</code> on a Node template, or <code>{{ drupal_view(\'ai_related_content\', \'default_block\', node.id) }}</code> to specify the current node ID. You can also revert the View back to its original state below.', [
              '@view_name' => $view_config_name,
              '@edit_link' => $url->toString(),
            ]),
          ],
        ],
      ];
    }
    else {
      $form['description'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [
            $this->t('This tool generates or resets a view called "AI Related Content" (<code>@view_name</code>) based on the selected index. It defaults to public access for all users. Review your own site\'s configuration to ensure only related content users are allowed to see are shown.', [
              '@view_name' => $view_config_name,
            ]),
          ],
        ],
      ];
    }

    // Details element containing the configuration options and button.
    $form['view_config_details'] = [
      '#type' => 'details',
      '#title' => $view_exists ? $this->t('Reset View Configuration') : $this->t('Create View Configuration'),
      '#open' => !$view_exists,
      '#weight' => 0,
    ];

    // Add a warning if the view already exists.
    if ($view_exists) {
      $form['view_config_details']['warning'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [
            $this->t('The "@view_name" view already exists. Submitting this form will <strong>reset the view configuration</strong> to the default template using the selected index, potentially overwriting any customizations you have made.', [
              '@view_name' => $view_config_name,
            ]),
          ],
        ],
      ];
    }

    $form['view_config_details']['search_api_index'] = [
      '#type' => 'select',
      '#title' => $this->t('Search API Index'),
      '#options' => $compatible_indexes,
      '#required' => TRUE,
      '#description' => $this->t('Choose the index to use for finding related content.'),
    ];

    $form['view_config_details']['search_api_index_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Search API Source Vectors Index'),
      '#options' => $compatible_indexes,
      '#required' => TRUE,
      '#description' => $this->t('Rather than generating new vectors, although cacheable, vectors can be retrieved from already generated ones by querying the vector database. This does not necessarily need to be the same index so long as the dimensions are the same and the Drupal Node ID exists. If not found, new vectors will be generated by calling the LLM embedding endpoint. It is useful to use a separate index if your index typically has many chunks per item. Then you likely want a separate index that has a giant chunk size so the entire Node is part of a single vector that represents the content as a whole to find related content for. When multiple chunks are found, an average pooling strategy is applied so the average of all chunks from the entity are used to find related content. You must verify the dimensions are the same or expect poor related content relevancy.'),
    ];

    $form['view_config_details']['actions']['#type'] = 'actions';
    $form['view_config_details']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $view_exists ? $this->t('Reset View') : $this->t('Create View'),
    ];

    // Add danger styling if resetting the view.
    if ($view_exists) {
      $form['view_config_details']['actions']['submit']['#attributes']['class'][] = 'button--danger';
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $selected_index_id = $form_state->getValue('search_api_index');
    $selected_index_source_id = $form_state->getValue('search_api_index_source');
    $module_path = $this->extensionPathResolver->getPath('module', 'ai_related_content');
    $template_file_path = $module_path . '/config/template-do-not-install/views.view.ai_related_content.template.yml';
    $view_config_name = 'views.view.ai_related_content';

    if (!file_exists($template_file_path)) {
      $this->messenger()->addError($this->t('View template file not found at %path.', ['%path' => $template_file_path]));
      $this->logger->error('AI Related Content view template file missing: @path', ['@path' => $template_file_path]);
      return;
    }

    try {
      $template_content = file_get_contents($template_file_path);
      if ($template_content === FALSE) {
        throw new \RuntimeException('Could not read template file content.');
      }

      // Replace the placeholder with the selected index machine name.
      $modified_yaml = str_replace('INDEX_NAME_HERE', $selected_index_id, $template_content);
      $modified_yaml = str_replace('INDEX_SOURCE_NAME_HERE', $selected_index_source_id, $modified_yaml);

      // Decode the modified YAML into a configuration array.
      $view_config_array = Yaml::decode($modified_yaml);

      if (isset($view_config_array['display']['default']['display_options']['row']['options']['view_modes']['entity:node'])) {
        $node_modes = &$view_config_array['display']['default']['display_options']['row']['options']['view_modes']['entity:node'];
        $node_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();

        // Set the default view mode for all node types stored in the current
        // site.
        /** @var \Drupal\node\Entity\NodeType $node_type */
        foreach ($node_types as $node_type_id => $node_type) {
          $node_modes[$node_type_id] = ':default';
        }
      }

      if (!is_array($view_config_array)) {
        throw new \RuntimeException('Failed to decode YAML from template after replacement.');
      }

      // Get the editable configuration object for the view.
      $config = $this->configFactory()->getEditable($view_config_name);

      // Set the data and save the configuration.
      $config->setData($view_config_array)->save();

      $this->messenger()->addStatus($this->t('The AI Related Content view configuration (@view_name) has been successfully created/updated using index @index.', [
        '@view_name' => $view_config_name,
        '@index' => $selected_index_id,
      ]));
      $this->logger->info('AI Related Content view configuration updated using index @index.', ['@index' => $selected_index_id]);

    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred while creating/updating the view configuration: %message', ['%message' => $e->getMessage()]));
      $this->logger->error('Failed to create/update AI Related Content view configuration: @message', ['@message' => $e->getMessage()]);
    }
  }

}
