<?php

namespace Drupal\ai_report\Form;

use Drupal\ai\Enum\AiModelCapability;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * AI Report form.
 */
class AiReportForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * AI Provider service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->database = $container->get('database');
    $instance->providerManager = $container->get('ai.provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_report_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Check access permission.
    if (!$this->currentUser()->hasPermission('access ai reports')) {
      $form['access_denied'] = [
        '#markup' => '<p>' . $this->t('You do not have permission to access AI Reports.') . '</p>',
      ];
      return $form;
    }

    // Attach the JavaScript library.
    $form['#attached']['library'][] = 'ai_report/ai_report';

    // Main query textarea.
    $form['query'] = [
      '#type' => 'textarea',
      '#title' => $this->t('What data do you want to see'),
      '#description' => $this->t('Enter your query here.'),
      '#rows' => 3,
      '#attributes' => [
        'class' => ['ai-report-query'],
        'placeholder' => $this->t('Count how many nodes of each type exist'),
      ],
    ];

    // Submit button with AJAX.
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Make report'),
      '#ajax' => [
        'callback' => '::ajaxSubmitCallback',
        'wrapper' => 'ai-report-results-wrapper',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating report...'),
        ],
      ],
    ];

    // Results container (AJAX wrapper).
    $form['results_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ai-report-results-wrapper'],
      '#weight' => 100,
    ];

    // Display results if form has been submitted.
    if ($form_state->has('results')) {
      $results = $form_state->get('results');

      $form['results_wrapper']['query_results'] = [
        '#type' => 'details',
        '#title' => $this->t('Query Results'),
        '#open' => TRUE,
      ];

      $form['results_wrapper']['query_results']['content'] = [
        '#markup' => $results['example_query'],
      ];

      $form['results_wrapper']['ai_response'] = [
        '#type' => 'details',
        '#title' => $this->t('AI Response'),
        '#open' => FALSE,
      ];

      $form['results_wrapper']['ai_response']['content'] = [
        '#markup' => '<pre>' . htmlspecialchars($results['ai_response'] ?? '') . '</pre>',
      ];

      $form['results_wrapper']['database_structure'] = [
        '#type' => 'details',
        '#title' => $this->t('Database Structure'),
        '#open' => FALSE,
      ];

      $form['results_wrapper']['database_structure']['content'] = [
        '#markup' => $results['database_structure'],
      ];

      $form['results_wrapper']['processed_prompt'] = [
        '#type' => 'details',
        '#title' => $this->t('Processed Prompt (sent to AI)'),
        '#open' => FALSE,
      ];

      $form['results_wrapper']['processed_prompt']['content'] = [
        '#markup' => '<pre>' . htmlspecialchars($results['processed_prompt']) . '</pre>',
      ];
    }

    return $form;
  }

  /**
   * AJAX callback for form submission.
   */
  public function ajaxSubmitCallback(array &$form, FormStateInterface $form_state) {
    return $form['results_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get user query from form.
    $user_query = $form_state->getValue('query');

    // Get configuration values.
    $config = $this->config('ai_report.settings');
    $prompt_template = $config->get('prompt');
    $ai_model = $config->get('ai_model');

    // Get database structure (plain text for AI).
    $database_structure_text = $this->getDatabaseStructureText();

    // Get content types.
    $content_types_text = $this->getContentTypes();

    // Replace placeholders in the prompt.
    $processed_prompt = str_replace(
      ['[actual_user_question]', '[database_structure]', '[contenttypes]'],
      [$user_query, $database_structure_text, $content_types_text],
      $prompt_template
    );

    // Get provider and model.
    if ($ai_model === '') {
      $default_provider = $this->providerManager->getDefaultProviderForOperationType('chat');
      $model_id = $default_provider['model_id'];
      $provider_id = $default_provider['provider_id'];
    }
    else {
      $parts = explode('__', $ai_model);
      $provider_id = $parts[0];
      $model_id = $parts[1];
    }

    // Validate we have provider and model.
    if (empty($provider_id) || empty($model_id)) {
      $this->messenger()->addError($this->t('No AI provider/model configured. Please configure an AI provider in the Configuration section or at /admin/config/ai.'));
      $form_state->setRebuild(TRUE);
      return;
    }

    // Call AI to get SQL query.
    $sql_query = '';
    $ai_response = '';
    try {
      $ai_response = $this->callAi($processed_prompt, $provider_id, $model_id);
      // Extract SQL query from AI response.
      $sql_query = $this->extractSqlQuery($ai_response);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error calling AI: @error', ['@error' => $e->getMessage()]));
    }

    // Get database structure (HTML for display).
    $database_structure_html = $this->getDatabaseStructure();

    // Execute the AI-generated SQL query.
    $query_results = '';
    if (!empty($sql_query)) {
      // Check that query starts with SELECT.
      $trimmed_query = trim($sql_query);
      if (!preg_match('/^SELECT\s+/i', $trimmed_query)) {
        $query_results = '<p class="error">' . $this->t('Query must start with SELECT. Only SELECT queries are allowed.') . '</p>';
      }
      else {
        $query_results = $this->executeAiQuery($sql_query);
      }
    }
    else {
      $query_results = '<p class="error">' . $this->t('No valid SQL query was generated by the AI.') . '</p>';
    }

    // Store results in form state to display them.
    $form_state->set('results', [
      'database_structure' => $database_structure_html,
      'example_query' => $query_results,
      'processed_prompt' => $processed_prompt,
      'ai_response' => $ai_response,
      'sql_query' => $sql_query,
    ]);

    // Rebuild the form to display results.
    $form_state->setRebuild(TRUE);
  }

  /**
   * Get database structure (tables and columns).
   *
   * @return string
   *   HTML formatted database structure.
   */
  protected function getDatabaseStructure() {
    $output = '<h3>' . $this->t('Database Tables and Columns') . '</h3>';

    // Get all tables.
    $tables = $this->database->schema()->findTables('%');

    $output .= '<table class="table table-bordered">';
    $output .= '<thead><tr><th>' . $this->t('Table') . '</th><th>' . $this->t('Columns') . '</th></tr></thead>';
    $output .= '<tbody>';

    foreach ($tables as $table) {
      // Get columns for this table.
      $columns = $this->database->query("DESCRIBE {" . $table . "}")->fetchAll();
      $column_names = [];
      foreach ($columns as $column) {
        $column_names[] = $column->Field;
      }

      $output .= '<tr>';
      $output .= '<td><strong>' . htmlspecialchars($table) . '</strong></td>';
      $output .= '<td>' . htmlspecialchars(implode(', ', $column_names)) . '</td>';
      $output .= '</tr>';
    }

    $output .= '</tbody></table>';

    return $output;
  }

  /**
   * Get content types.
   *
   * @return string
   *   Comma-separated list of content types.
   */
  protected function getContentTypes() {
    try {
      // Query to get all content types from node_type table.
      $query = $this->database->select('node_field_data', 'nfd');
      $query->fields('nfd', ['type']);
      $results = $query->execute()->fetchCol();

      if (empty($results)) {
        return 'none';
      }

      return implode(', ', $results);
    }
    catch (\Exception $e) {
      return 'error fetching content types';
    }
  }

  /**
   * Get database structure as plain text (for AI prompt).
   *
   * @return string
   *   Plain text formatted database structure.
   */
  protected function getDatabaseStructureText() {
    $output = '';

    // Get all tables.
    $tables = $this->database->schema()->findTables('%');

    foreach ($tables as $table) {
      // Get columns for this table.
      $columns = $this->database->query("DESCRIBE {" . $table . "}")->fetchAll();
      $column_names = [];
      foreach ($columns as $column) {
        $column_names[] = $column->Field . ' (' . $column->Type . ')';
      }

      $output .= "Table: {$table}\n";
      $output .= "Columns: " . implode(', ', $column_names) . "\n\n";
    }

    return $output;
  }

  /**
   * Execute example user query.
   *
   * @return string
   *   HTML formatted query results.
   */
  protected function getExampleUserQuery() {
    $output = '<h3>' . $this->t('Users Query Results') . '</h3>';

    try {
      // Select all users with their fields, sorted chronologically.
      $query = $this->database->select('users_field_data', 'u');
      $query->fields('u');
      $query->orderBy('u.created', 'ASC');
      $results = $query->execute()->fetchAll();

      if (empty($results)) {
        return $output . '<p>' . $this->t('No users found.') . '</p>';
      }

      // Build HTML table.
      $output .= '<table class="table table-bordered table-striped">';

      // Header row.
      $output .= '<thead><tr>';
      $first_row = (array) $results[0];
      foreach (array_keys($first_row) as $field) {
        $output .= '<th>' . htmlspecialchars($field) . '</th>';
      }
      $output .= '</tr></thead>';

      // Data rows.
      $output .= '<tbody>';
      foreach ($results as $row) {
        $output .= '<tr>';
        foreach ((array) $row as $value) {
          $output .= '<td>' . htmlspecialchars($value ?? '') . '</td>';
        }
        $output .= '</tr>';
      }
      $output .= '</tbody>';

      $output .= '</table>';

      $output .= '<p>' . $this->t('Total users: @count', ['@count' => count($results)]) . '</p>';

    }
    catch (\Exception $e) {
      $output .= '<p class="error">' . $this->t('Error executing query: @error', ['@error' => $e->getMessage()]) . '</p>';
    }

    return $output;
  }

  /**
   * Call AI with the processed prompt.
   *
   * @param string $prompt
   *   The prompt to send to AI.
   * @param string $provider_id
   *   The provider ID.
   * @param string $model_id
   *   The model ID.
   *
   * @return string
   *   The AI response.
   */
  protected function callAi(string $prompt, string $provider_id, string $model_id): string {
    // Get the provider instance.
    $provider = $this->providerManager->createInstance($provider_id);

    // Create chat message.
    $message = new ChatMessage('user', $prompt);
    $chat_input = new ChatInput([$message]);

    // Call the AI.
    $response = $provider->chat($chat_input, $model_id);

    // Get the response text from the normalized ChatMessage.
    return $response->getNormalized()->getText();
  }

  /**
   * Check for illegal SQL keywords.
   *
   * @param string $sql_query
   *   The SQL query to check.
   *
   * @return string|false
   *   The illegal keyword found, or FALSE if none.
   */
  protected function checkIllegalKeywords(string $sql_query): string|false {
    // List of illegal keywords that would modify data.
    $illegal_keywords = ['INSERT', 'UPDATE', 'DELETE', 'DROP', 'TRUNCATE', 'ALTER', 'CREATE', 'REPLACE'];

    // Convert query to uppercase for case-insensitive checking.
    $sql_upper = strtoupper($sql_query);

    // Check each illegal keyword.
    foreach ($illegal_keywords as $keyword) {
      if (str_contains($sql_upper, $keyword)) {
        return $keyword;
      }
    }

    return FALSE;
  }

  /**
   * Extract SQL query from AI response.
   *
   * @param string $response
   *   The AI response.
   *
   * @return string
   *   The extracted SQL query.
   */
  protected function extractSqlQuery(string $response): string {
    // Remove markdown code blocks if present.
    $response = trim($response);

    // Try to extract SQL from markdown code block.
    if (preg_match('/```sql\s*(.*?)\s*```/s', $response, $matches)) {
      return trim($matches[1]);
    }

    // Try to extract from generic code block.
    if (preg_match('/```\s*(.*?)\s*```/s', $response, $matches)) {
      return trim($matches[1]);
    }

    // If no code block, assume the entire response is the SQL query.
    return trim($response);
  }

  /**
   * Execute AI-generated SQL query.
   *
   * @param string $sql_query
   *   The SQL query to execute.
   *
   * @return string
   *   HTML formatted query results.
   */
  protected function executeAiQuery(string $sql_query): string {
    $output = '<h3>' . $this->t('Results') . '</h3>';

    try {
      // Execute the query.
      $results = $this->database->query($sql_query)->fetchAll();

      if (empty($results)) {
        return $output . '<p>' . $this->t('No results found.') . '</p>';
      }

      // Build HTML table.
      $output .= '<table class="table table-bordered table-striped">';

      // Header row.
      $output .= '<thead><tr>';
      $first_row = (array) $results[0];
      foreach (array_keys($first_row) as $field) {
        $output .= '<th>' . htmlspecialchars($field) . '</th>';
      }
      $output .= '</tr></thead>';

      // Data rows.
      $output .= '<tbody>';
      foreach ($results as $row) {
        $output .= '<tr>';
        foreach ((array) $row as $value) {
          $output .= '<td>' . htmlspecialchars($value ?? '') . '</td>';
        }
        $output .= '</tr>';
      }
      $output .= '</tbody>';

      $output .= '</table>';

      $output .= '<p>' . $this->t('Total results: @count', ['@count' => count($results)]) . '</p>';

    }
    catch (\Exception $e) {
      $output .= '<p class="error">' . $this->t('Error executing query: @error', ['@error' => $e->getMessage()]) . '</p>';
    }

    return $output;
  }

}
