<?php

namespace Drupal\ai_report\Form;

use Drupal\ai\Enum\AiModelCapability;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AI Report settings.
 */
class AiReportSettingsForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_report.settings';

  /**
   * AI Provider service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->providerManager = $container->get('ai.provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_report_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Check access permission.
    if (!$this->currentUser()->hasPermission('administer ai reports')) {
      $form['access_denied'] = [
        '#markup' => '<p>' . $this->t('You do not have permission to administer AI Reports.') . '</p>',
      ];
      return $form;
    }

    $config = $this->config(static::CONFIG_NAME);

    // AI model selection.
    $models = $this->providerManager->getSimpleProviderModelOptions('chat', TRUE, TRUE, [
      AiModelCapability::ChatWithImageVision,
    ]);

    $form['ai_model'] = [
      '#title' => $this->t('AI provider/model'),
      '#type' => 'select',
      '#options' => $models,
      '#default_value' => $config->get('ai_model') ?? '',
      '#empty_option' => $this->t('Use Default Model'),
      '#description' => $this->t('AI model to use for generating the report.'),
    ];

    // Prompt configuration.
    $default_prompt = 'INSTRUCTION:
------------
You are a AI assistant for analyzing user\'s questions. You answer with
- A list (sql query) formatted as a HTML table
- A list (sql query) formatted as a HTML table + in combination with a graph (made with js).
you take into account the Database structure (see DATABASE STRUCTURE)
you try to answer the question of the user (see USER_QUESTION)
You only output the SQL query needed to find the relevant data. nothing else, no answers, no pleasantries, nothing else.

People sometimes use wrong content type names in their question. Check in the list of contenttypes to create your query correctly.

CRITICAL RULES:
1. ONLY use columns that exist in the tables shown in DATABASE STRUCTURE below
2. Check the exact column names and types for each table before writing your query
3. Do NOT assume column names - verify them in the structure first
4. For Drupal nodes: use node_field_data table for title, created, changed, uid, status, type
5. The node table only contains: nid, vid, type, uuid, langcode
6. When joining tables, ensure all columns in SELECT exist in the tables you\'re using

COMMON DRUPAL PATTERNS:
- For node data with title/dates: SELECT from node_field_data (has: nid, vid, type, langcode, title, uid, status, created, changed, etc.)
- For node body: JOIN node__body (has: entity_id, body_value, body_summary, body_format)
- For users: use users_field_data (has: uid, name, mail, created, changed, status, etc.)
- Always use the table alias in SELECT when joining multiple tables

FORMAT
---------
you return an executable SQL query, as a string nothing else
use a mysql compliant standard syntax. SQL:1999

VARIABLES
---------
content types: [contenttypes]

USER_QUESTION
------------
[actual_user_question]

DATABASE STRUCTURE
------------------
[database_structure]';

    $form['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt Configuration'),
      '#default_value' => $config->get('prompt') ?? $default_prompt,
      '#description' => $this->t('Configure the AI prompt template. Use [actual_user_question], [database_structure], and [contenttypes] as placeholders.'),
      '#rows' => 20,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::CONFIG_NAME)
      ->set('ai_model', $form_state->getValue('ai_model'))
      ->set('prompt', $form_state->getValue('prompt'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
