<?php

namespace Drupal\ai_revision_log\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Generates AI-powered revision log messages for content changes.
 *
 * @Action(
 *   id = "ai_revision_log_generate",
 *   label = @Translation("Generate AI revision log message"),
 *   type = "entity",
 *   category = @Translation("AI")
 * )
 */
class GenerateRevisionLogAction extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a GenerateRevisionLogAction object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AiProviderPluginManager $ai_provider_manager, LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->aiProviderManager = $ai_provider_manager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai.provider'),
      $container->get('logger.factory')->get('ai_revision_log')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    if (!$entity instanceof EntityInterface || !$entity instanceof RevisionableInterface) {
      return;
    }

    // Only process published entities.
    if (method_exists($entity, 'isPublished') && !$entity->isPublished()) {
      return;
    }

    $revision_ids = \Drupal::entityTypeManager()
      ->getStorage($entity->getEntityTypeId())
      ->revisionIds($entity);
    
    $count = count($revision_ids);
    if ($count < 2) {
      return;
    }

    $previous_rev_id = $revision_ids[$count - 2];
    $previous = \Drupal::entityTypeManager()
      ->getStorage($entity->getEntityTypeId())
      ->loadRevision($previous_rev_id);

    $changes = $this->detectChanges($entity, $previous);

    if (!empty($changes)) {
      $summary = $this->generateAiSummary($changes);
      if ($summary) {
        $entity->setRevisionLogMessage($summary);
        $entity->save();
      }
    }
  }

  /**
   * Detects changes between two entity revisions.
   *
   * @param \Drupal\Core\Entity\EntityInterface $current
   *   The current entity revision.
   * @param \Drupal\Core\Entity\EntityInterface $previous
   *   The previous entity revision.
   *
   * @return array
   *   An array of detected changes.
   */
  protected function detectChanges(EntityInterface $current, EntityInterface $previous) {
    $changes = [];
    $ignored_fields = [
      'vid', 'revision_timestamp', 'revision_uid', 'revision_log',
      'changed', 'uuid', 'langcode', 'default_langcode',
    ];

    foreach ($current->getFields() as $field_name => $field) {
      // Skip internal Drupal fields.
      if (in_array($field_name, $ignored_fields)) {
        continue;
      }

      // Skip empty fields that were also empty before.
      if ($field->isEmpty() && $previous->get($field_name)->isEmpty()) {
        continue;
      }

      $current_value = $field->getString();
      $previous_value = $previous->get($field_name)->getString();

      if ($current_value !== $previous_value) {
        $field_definition = $current->getFieldDefinition($field_name);
        $field_label = $field_definition ? $field_definition->getLabel() : $field_name;

        // Truncate long values for readability.
        $current_display = strlen($current_value) > 100 ? substr($current_value, 0, 100) . '...' : $current_value;
        $previous_display = strlen($previous_value) > 100 ? substr($previous_value, 0, 100) . '...' : $previous_value;

        $changes[] = sprintf(
          "%s changed from '%s' to '%s'",
          $field_label,
          $previous_display,
          $current_display
        );
      }
    }

    return $changes;
  }

  /**
   * Generates an AI summary of the changes.
   *
   * @param array $changes
   *   An array of detected changes.
   *
   * @return string|null
   *   The AI-generated summary or NULL on failure.
   */
  protected function generateAiSummary(array $changes) {
    $changes_text = implode("\n", $changes);

    try {
      // Get default AI provider configuration.
      $defaults = $this->aiProviderManager->getDefaultProviderForOperationType('chat');
      $provider_id = $defaults['provider_id'] ?? 'openai';
      $model_id = $defaults['model_id'] ?? 'gpt-3.5-turbo';

      // Get the AI provider instance.
      $provider = $this->aiProviderManager->createInstance($provider_id);

      if (!$provider) {
        throw new \Exception("AI provider '{$provider_id}' not found or not configured.");
      }

      // Create chat input.
      $chat_input = new ChatInput([
        new ChatMessage('system', 'Summarize the following content changes concisely. Focus only on meaningful content changes. Provide a brief, professional revision log message suitable for content management.'),
        new ChatMessage('user', $changes_text),
      ]);

      // Send the message to the AI provider.
      $response = $provider->chat($chat_input, $model_id);

      if ($response && $response->getNormalized()) {
        return trim($response->getNormalized()->getText());
      } else {
        throw new \Exception('No response received from the AI provider.');
      }
    }
    catch (\Exception $e) {
      $this->logger->error('AI revision log generation failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    if ($object instanceof EntityInterface) {
      return $object->access('update', $account, $return_as_object);
    }
    return FALSE;
  }

}