<?php

namespace Drupal\ai_revision_log\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Generates AI-powered revision log messages for content changes.
 *
 * @Action(
 *   id = "ai_revision_log_generate",
 *   label = @Translation("Generate AI revision log message"),
 *   type = "entity",
 *   category = @Translation("AI")
 * )
 */
class GenerateRevisionLogAction extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a GenerateRevisionLogAction object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AiProviderPluginManager $ai_provider_manager, LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->aiProviderManager = $ai_provider_manager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai.provider'),
      $container->get('logger.factory')->get('ai_revision_log')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    if (!$entity instanceof EntityInterface || !$entity instanceof RevisionableInterface) {
      return;
    }

    // Only process published entities.
    if (method_exists($entity, 'isPublished') && !$entity->isPublished()) {
      return;
    }

    $revision_ids = \Drupal::entityTypeManager()
      ->getStorage($entity->getEntityTypeId())
      ->revisionIds($entity);
    
    $count = count($revision_ids);
    if ($count < 2) {
      return;
    }

    $previous_rev_id = $revision_ids[$count - 2];
    $previous = \Drupal::entityTypeManager()
      ->getStorage($entity->getEntityTypeId())
      ->loadRevision($previous_rev_id);

    $changes = $this->detectChanges($entity, $previous);

    if (!empty($changes)) {
      $summary = $this->generateAiSummary($changes);
      if ($summary) {
        $entity->setRevisionLogMessage($summary);
        $entity->save();
      }
    }
  }

  /**
   * Detects changes between two entity revisions.
   *
   * @param \Drupal\Core\Entity\EntityInterface $current
   *   The current entity revision.
   * @param \Drupal\Core\Entity\EntityInterface $previous
   *   The previous entity revision.
   *
   * @return array
   *   An array of detected changes.
   */
  protected function detectChanges(EntityInterface $current, EntityInterface $previous) {
    $changes = [];
    $ignored_fields = [
      'vid', 'revision_timestamp', 'revision_uid', 'revision_log',
      'changed', 'uuid', 'langcode', 'default_langcode',
    ];

    foreach ($current->getFields() as $field_name => $field) {
      // Skip internal Drupal fields.
      if (in_array($field_name, $ignored_fields)) {
        continue;
      }

      // Skip empty fields that were also empty before.
      if ($field->isEmpty() && $previous->get($field_name)->isEmpty()) {
        continue;
      }

      $field_definition = $current->getFieldDefinition($field_name);
      $field_type = $field_definition ? $field_definition->getType() : '';
      $field_label = $field_definition ? $field_definition->getLabel() : $field_name;

      // Handle different field types appropriately.
      $change_description = $this->analyzeFieldChange($field, $previous->get($field_name), $field_type, $field_label);
      
      if ($change_description) {
        $changes[] = $change_description;
      }
    }

    return $changes;
  }

  /**
   * Analyzes changes in a specific field based on its type.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $current_field
   *   The current field value.
   * @param \Drupal\Core\Field\FieldItemListInterface $previous_field
   *   The previous field value.
   * @param string $field_type
   *   The field type.
   * @param string $field_label
   *   The human-readable field label.
   *
   * @return string|null
   *   A description of the change, or NULL if no significant change.
   */
  protected function analyzeFieldChange($current_field, $previous_field, $field_type, $field_label) {
    // Handle paragraph fields.
    if ($field_type === 'entity_reference_revisions') {
      return $this->analyzeParagraphChanges($current_field, $previous_field, $field_label);
    }

    // Handle Layout Builder fields.
    if ($field_type === 'layout_section') {
      return $this->analyzeLayoutBuilderChanges($current_field, $previous_field, $field_label);
    }

    // Handle entity reference fields.
    if ($field_type === 'entity_reference') {
      return $this->analyzeEntityReferenceChanges($current_field, $previous_field, $field_label);
    }

    // Handle image/media fields.
    if (in_array($field_type, ['image', 'file'])) {
      return $this->analyzeFileChanges($current_field, $previous_field, $field_label);
    }

    // Default handling for other field types.
    $current_value = $current_field->getString();
    $previous_value = $previous_field->getString();

    if ($current_value !== $previous_value) {
      // Truncate long values for readability.
      $current_display = strlen($current_value) > 100 ? substr($current_value, 0, 100) . '...' : $current_value;
      $previous_display = strlen($previous_value) > 100 ? substr($previous_value, 0, 100) . '...' : $previous_value;

      return sprintf(
        "%s changed from '%s' to '%s'",
        $field_label,
        $previous_display,
        $current_display
      );
    }

    return NULL;
  }

  /**
   * Analyzes changes in paragraph fields.
   */
  protected function analyzeParagraphChanges($current_field, $previous_field, $field_label) {
    $current_count = $current_field->count();
    $previous_count = $previous_field->count();

    $changes = [];

    // Handle paragraph additions/removals.
    if ($current_count > $previous_count) {
      $added = $current_count - $previous_count;
      $changes[] = sprintf("Added %d new paragraph(s)", $added);
    } elseif ($current_count < $previous_count) {
      $removed = $previous_count - $current_count;
      $changes[] = sprintf("Removed %d paragraph(s)", $removed);
    }

    // Check for content changes in existing paragraphs.
    $min_count = min($current_count, $previous_count);
    for ($i = 0; $i < $min_count; $i++) {
      $current_paragraph = $current_field->get($i)->entity;
      $previous_paragraph = $previous_field->get($i)->entity;

      if ($current_paragraph && $previous_paragraph) {
        $paragraph_type = $current_paragraph->bundle();
        $detailed_changes = $this->analyzeParagraphFieldChanges($current_paragraph, $previous_paragraph);
        
        if (!empty($detailed_changes)) {
          foreach ($detailed_changes as $change) {
            $changes[] = sprintf("%s section: %s", ucfirst(str_replace('_', ' ', $paragraph_type)), $change);
          }
        }
      }
    }

    if (!empty($changes)) {
      return implode("\n", $changes);
    }

    return NULL;
  }

  /**
   * Analyzes changes in Layout Builder fields.
   */
  protected function analyzeLayoutBuilderChanges($current_field, $previous_field, $field_label) {
    $current_sections = $current_field->getValue();
    $previous_sections = $previous_field->getValue();

    if (count($current_sections) !== count($previous_sections)) {
      return sprintf("%s: layout sections changed from %d to %d", $field_label, count($previous_sections), count($current_sections));
    }

    // Check for component changes within sections.
    $layout_changes = [];
    foreach ($current_sections as $index => $current_section) {
      if (isset($previous_sections[$index])) {
        $current_components = count($current_section['section']->getComponents());
        $previous_components = count($previous_sections[$index]['section']->getComponents());

        if ($current_components !== $previous_components) {
          $layout_changes[] = sprintf("section %d components changed", $index + 1);
        }
      }
    }

    if (!empty($layout_changes)) {
      return sprintf("%s: %s", $field_label, implode(', ', $layout_changes));
    }

    return NULL;
  }

  /**
   * Analyzes changes in entity reference fields.
   */
  protected function analyzeEntityReferenceChanges($current_field, $previous_field, $field_label) {
    $current_ids = array_column($current_field->getValue(), 'target_id');
    $previous_ids = array_column($previous_field->getValue(), 'target_id');

    $added = array_diff($current_ids, $previous_ids);
    $removed = array_diff($previous_ids, $current_ids);

    $changes = [];
    if (!empty($added)) {
      $changes[] = sprintf("%d added", count($added));
    }
    if (!empty($removed)) {
      $changes[] = sprintf("%d removed", count($removed));
    }

    if (!empty($changes)) {
      return sprintf("%s: %s", $field_label, implode(', ', $changes));
    }

    return NULL;
  }

  /**
   * Analyzes changes in file/image fields.
   */
  protected function analyzeFileChanges($current_field, $previous_field, $field_label) {
    $current_files = array_column($current_field->getValue(), 'target_id');
    $previous_files = array_column($previous_field->getValue(), 'target_id');

    if ($current_files !== $previous_files) {
      return sprintf("%s: files changed", $field_label);
    }

    return NULL;
  }

  /**
   * Analyzes field changes within individual paragraphs.
   */
  protected function analyzeParagraphFieldChanges($current_paragraph, $previous_paragraph) {
    $changes = [];
    $ignored_fields = [
      'id', 'uuid', 'type', 'parent_id', 'parent_type', 'parent_field_name',
      'revision_id', 'langcode', 'default_langcode', 'revision_default',
      'revision_translation_affected', 'created', 'behavior_settings'
    ];

    foreach ($current_paragraph->getFields() as $field_name => $field) {
      // Skip internal fields.
      if (in_array($field_name, $ignored_fields)) {
        continue;
      }

      $previous_field = $previous_paragraph->get($field_name);
      
      // Skip if both fields are empty.
      if ($field->isEmpty() && $previous_field->isEmpty()) {
        continue;
      }

      $field_definition = $current_paragraph->getFieldDefinition($field_name);
      $field_label = $field_definition ? $field_definition->getLabel() : $field_name;
      $field_type = $field_definition ? $field_definition->getType() : '';

      // Analyze the specific field change.
      $change_description = $this->analyzeSpecificFieldChange($field, $previous_field, $field_type, $field_label);
      
      if ($change_description) {
        $changes[] = $change_description;
      }
    }

    return $changes;
  }

  /**
   * Analyzes specific field changes with more detail.
   */
  protected function analyzeSpecificFieldChange($current_field, $previous_field, $field_type, $field_label) {
    $current_value = $current_field->getString();
    $previous_value = $previous_field->getString();

    if ($current_value === $previous_value) {
      return NULL;
    }

    // Handle different field types with specific formatting.
    switch ($field_type) {
      case 'string':
      case 'string_long':
        if (empty($previous_value)) {
          return sprintf("%s set to '%s'", $field_label, $this->truncateText($current_value, 60));
        } elseif (empty($current_value)) {
          return sprintf("%s cleared", $field_label);
        } else {
          return sprintf("%s updated from '%s' to '%s'", 
            $field_label, 
            $this->truncateText($previous_value, 60), 
            $this->truncateText($current_value, 60)
          );
        }

      case 'text':
      case 'text_long':
      case 'text_with_summary':
        if (empty($previous_value)) {
          return sprintf("%s added", $field_label);
        } elseif (empty($current_value)) {
          return sprintf("%s removed", $field_label);
        } else {
          // For text fields, show a preview of the change
          $current_preview = $this->truncateText(strip_tags($current_value), 100);
          $previous_preview = $this->truncateText(strip_tags($previous_value), 100);
          
          if ($current_preview !== $previous_preview) {
            return sprintf("%s updated", $field_label);
          }
          return sprintf("%s modified", $field_label);
        }

      case 'image':
      case 'file':
        return sprintf("%s updated", $field_label);

      case 'link':
        $current_url = $current_field->first() ? $current_field->first()->get('uri')->getValue() : '';
        $previous_url = $previous_field->first() ? $previous_field->first()->get('uri')->getValue() : '';
        
        if ($current_url !== $previous_url) {
          return sprintf("%s URL changed", $field_label);
        }
        return sprintf("%s updated", $field_label);

      case 'boolean':
        $current_bool = $current_field->value ? 'enabled' : 'disabled';
        $previous_bool = $previous_field->value ? 'enabled' : 'disabled';
        return sprintf("%s %s (was %s)", $field_label, $current_bool, $previous_bool);

      default:
        return sprintf("%s modified", $field_label);
    }
  }

  /**
   * Truncates text to specified length with ellipsis.
   */
  protected function truncateText($text, $length = 100) {
    return strlen($text) > $length ? substr($text, 0, $length) . '...' : $text;
  }

  /**
   * Gets a summary of a paragraph entity's content.
   */
  protected function getParagraphSummary($paragraph) {
    if (!$paragraph) {
      return '';
    }

    $summary_parts = [];
    foreach ($paragraph->getFields() as $field_name => $field) {
      if (!in_array($field_name, ['id', 'uuid', 'type', 'parent_id', 'parent_type', 'parent_field_name'])) {
        if (!$field->isEmpty()) {
          $summary_parts[] = substr($field->getString(), 0, 50);
        }
      }
    }

    return implode(' | ', $summary_parts);
  }

  /**
   * Generates an AI summary of the changes.
   *
   * @param array $changes
   *   An array of detected changes.
   *
   * @return string|null
   *   The AI-generated summary or NULL on failure.
   */
  protected function generateAiSummary(array $changes) {
    $changes_text = implode("\n", $changes);

    try {
      // Get default AI provider configuration.
      $defaults = $this->aiProviderManager->getDefaultProviderForOperationType('chat');
      $provider_id = $defaults['provider_id'] ?? 'openai';
      $model_id = $defaults['model_id'] ?? 'gpt-3.5-turbo';

      // Get the AI provider instance.
      $provider = $this->aiProviderManager->createInstance($provider_id);

      if (!$provider) {
        throw new \Exception("AI provider '{$provider_id}' not found or not configured.");
      }

      // Create chat input.
      $chat_input = new ChatInput([
        new ChatMessage('system', 'Create a well-organized revision log from the following content changes. Format as numbered sections:

**Content Changes:**
1. [Change description]
2. [Change description]
3. [Change description]

Guidelines:
- Group related changes logically
- Use clear, specific descriptions
- Remove redundant "paragraph (item X)" references - just mention the content type
- Focus on what actually changed, not technical details
- Keep descriptions concise but meaningful
- Use action words: "Updated", "Added", "Removed", "Modified"
- For text changes, mention key words or concepts rather than character counts'),
        new ChatMessage('user', $changes_text),
      ]);

      // Send the message to the AI provider.
      $response = $provider->chat($chat_input, $model_id);

      if ($response && $response->getNormalized()) {
        return trim($response->getNormalized()->getText());
      } else {
        throw new \Exception('No response received from the AI provider.');
      }
    }
    catch (\Exception $e) {
      $this->logger->error('AI revision log generation failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    if ($object instanceof EntityInterface) {
      return $object->access('update', $account, $return_as_object);
    }
    return FALSE;
  }

}