<?php

namespace Drupal\Tests\ai_revision_log\Unit;

use Drupal\ai_revision_log\Plugin\Action\GenerateRevisionLogAction;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\AiProviderInterface;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Psr\Log\LoggerInterface;

/**
 * Tests for the GenerateRevisionLogAction.
 *
 * @group ai_revision_log
 * @coversDefaultClass \Drupal\ai_revision_log\Plugin\Action\GenerateRevisionLogAction
 */
class GenerateRevisionLogActionTest extends UnitTestCase {

  /**
   * The action under test.
   *
   * @var \Drupal\ai_revision_log\Plugin\Action\GenerateRevisionLogAction
   */
  protected $action;

  /**
   * Mock AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $aiProviderManager;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->aiProviderManager = $this->createMock(AiProviderPluginManager::class);
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);

    // Mock the entity type manager service.
    $container = $this->createMock('\Symfony\Component\DependencyInjection\ContainerInterface');
    $container->method('get')
      ->willReturnMap([
        ['entity_type.manager', 1, $this->entityTypeManager],
      ]);
    \Drupal::setContainer($container);

    $this->action = new GenerateRevisionLogAction(
      [],
      'ai_revision_log_generate',
      ['type' => 'entity'],
      $this->aiProviderManager,
      $this->logger
    );
  }

  /**
   * Tests execute method with non-revisionable entity.
   *
   * @covers ::execute
   */
  public function testExecuteWithNonRevisionableEntity() {
    $entity = $this->createMock(EntityInterface::class);
    
    // The action should return early for non-revisionable entities.
    $this->action->execute($entity);
    
    // No assertions needed - we just verify no exceptions are thrown.
    $this->assertTrue(TRUE);
  }

  /**
   * Tests execute method with unpublished entity.
   *
   * @covers ::execute
   */
  public function testExecuteWithUnpublishedEntity() {
    $entity = $this->createMock(TestRevisionableEntity::class);
    $entity->method('isPublished')->willReturn(FALSE);
    
    // The action should return early for unpublished entities.
    $this->action->execute($entity);
    
    // No assertions needed - we just verify no exceptions are thrown.
    $this->assertTrue(TRUE);
  }

  /**
   * Tests execute method with insufficient revisions.
   *
   * @covers ::execute
   */
  public function testExecuteWithInsufficientRevisions() {
    $entity = $this->createMock(TestRevisionableEntity::class);
    $entity->method('isPublished')->willReturn(TRUE);
    $entity->method('getEntityTypeId')->willReturn('node');

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('revisionIds')->willReturn([1]); // Only one revision

    $this->entityTypeManager->method('getStorage')->willReturn($storage);

    // The action should return early with insufficient revisions.
    $this->action->execute($entity);
    
    // No assertions needed - we just verify no exceptions are thrown.
    $this->assertTrue(TRUE);
  }

  /**
   * Tests detectChanges method.
   *
   * @covers ::detectChanges
   */
  public function testDetectChanges() {
    $current_entity = $this->createMock(TestRevisionableEntity::class);
    $previous_entity = $this->createMock(TestRevisionableEntity::class);

    // Mock field for current entity.
    $current_field = $this->createMock(FieldItemListInterface::class);
    $current_field->method('isEmpty')->willReturn(FALSE);
    $current_field->method('getString')->willReturn('New title');

    // Mock field for previous entity.
    $previous_field = $this->createMock(FieldItemListInterface::class);
    $previous_field->method('isEmpty')->willReturn(FALSE);
    $previous_field->method('getString')->willReturn('Old title');

    // Mock field definition.
    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $field_definition->method('getLabel')->willReturn('Title');

    $current_entity->method('getFields')->willReturn(['title' => $current_field]);
    $current_entity->method('get')->willReturn($current_field);
    $current_entity->method('getFieldDefinition')->willReturn($field_definition);

    $previous_entity->method('get')->willReturn($previous_field);

    $reflection = new \ReflectionClass($this->action);
    $method = $reflection->getMethod('detectChanges');
    $method->setAccessible(TRUE);

    $changes = $method->invokeArgs($this->action, [$current_entity, $previous_entity]);

    $this->assertCount(1, $changes);
    $this->assertStringContainsString('Title changed from', $changes[0]);
  }

  /**
   * Tests generateAiSummary method with successful AI response.
   *
   * @covers ::generateAiSummary
   */
  public function testGenerateAiSummarySuccess() {
    $changes = ['Title changed from "Old" to "New"'];

    // Mock AI provider.
    $provider = $this->createMock(AiProviderInterface::class);
    $this->aiProviderManager->method('getDefaultProviderForOperationType')
      ->willReturn(['provider_id' => 'openai', 'model_id' => 'gpt-3.5-turbo']);
    $this->aiProviderManager->method('createInstance')->willReturn($provider);

    // Mock AI response.
    $chat_output = $this->createMock(ChatOutput::class);
    $normalized_response = $this->createMock(ChatMessage::class);
    $normalized_response->method('getText')->willReturn('Updated title content');
    
    $chat_output->method('getNormalized')->willReturn($normalized_response);
    $provider->method('chat')->willReturn($chat_output);

    $reflection = new \ReflectionClass($this->action);
    $method = $reflection->getMethod('generateAiSummary');
    $method->setAccessible(TRUE);

    $result = $method->invokeArgs($this->action, [$changes]);

    $this->assertEquals('Updated title content', $result);
  }

  /**
   * Tests generateAiSummary method with AI provider failure.
   *
   * @covers ::generateAiSummary
   */
  public function testGenerateAiSummaryFailure() {
    $changes = ['Title changed from "Old" to "New"'];

    $this->aiProviderManager->method('getDefaultProviderForOperationType')
      ->willReturn(['provider_id' => 'openai', 'model_id' => 'gpt-3.5-turbo']);
    $this->aiProviderManager->method('createInstance')->willReturn(NULL);

    $this->logger->expects($this->once())->method('error');

    $reflection = new \ReflectionClass($this->action);
    $method = $reflection->getMethod('generateAiSummary');
    $method->setAccessible(TRUE);

    $result = $method->invokeArgs($this->action, [$changes]);

    $this->assertNull($result);
  }

  /**
   * Tests access method.
   *
   * @covers ::access
   */
  public function testAccess() {
    $entity = $this->createMock(EntityInterface::class);
    $entity->method('access')->willReturn(TRUE);

    $result = $this->action->access($entity);

    $this->assertTrue($result);
  }

}

/**
 * Test interface that combines EntityInterface and RevisionableInterface.
 */
interface TestRevisionableEntity extends EntityInterface, RevisionableInterface {

  /**
   * Mock isPublished method.
   */
  public function isPublished();

}