<?php

namespace Drupal\ai_search_block_log\Controller;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
// Adjust namespace if necessary.
use Drupal\ai_search_block_log\AiSearchBlockLogHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * An example controller.
 */
class AiSearchBlockLogController extends ControllerBase {

  /**
   * Constructs a new AiSearchBlockLogController.
   *
   * @param \Drupal\ai_search_block_log\AiSearchBlockLogHelper $helper
   *   The log helper service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\ai\AiProviderPluginManager $aiProviderManager
   *   The AI provider manager.
   */
  public function __construct(
    protected AiSearchBlockLogHelper $helper,
    protected CacheBackendInterface $cache,
    protected Connection $database,
    protected AiProviderPluginManager $aiProviderManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): AiSearchBlockLogController {
    return new static(
      $container->get('ai_search_block_log.helper'),
      $container->get('cache.default'),
      $container->get('database'),
      $container->get('ai.provider')
    );
  }

  /**
   * Returns a renderable array for a test page.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response.
   */
  public function score(Request $request): JsonResponse {
    $logId = $request->request->get('log_id') ?? $request->query->get('log_id');
    if (empty($logId)) {
      throw new NotFoundHttpException('Log ID not found.');
    }

    $score = $request->request->get('score') ?? $request->query->get('score');
    $this->helper->update((int) $logId, ['score' => (int) $score]);
    if ($feedback = $request->request->get('feedback') ?? $request->query->get('feedback')) {
      $this->helper->update((int) $logId, ['feedback' => $feedback]);
    }

    return new JsonResponse([
      'response' => $this->t('Thank you for your feedback.'),
    ]);
  }

  /**
   * Display graphs overview page with key metrics only.
   *
   * @return array
   *   A renderable array.
   */
  public function graphsOverview(): array {
    // Generate cache key based on current day.
    $cache_key = 'ai_search_block_log:graphs_overview:' . date('Y-m-d');

    // Try to get cached data.
    $cache = $this->cache->get($cache_key);
    if ($cache && !empty($cache->data)) {
      return $cache->data;
    }

    $connection = $this->database;

    // Get current and previous month date range.
    $current_month_start = strtotime('first day of this month 00:00:00');
    $previous_month_start = strtotime('first day of last month 00:00:00');
    $current_month_end = time();

    // Get scoring stats.
    $scoring_stats = $this->getScoringStats($connection, $previous_month_start, $current_month_end);

    // Get monthly totals.
    $monthly_totals = $this->getMonthlyTotals($connection, $current_month_start);

    $build = [
      '#theme' => 'ai_search_block_log_graphs_overview',
      '#scoring_stats' => $scoring_stats,
      '#monthly_totals' => $monthly_totals,
      '#attached' => [
        'library' => [
          'ai_search_block_log/graphs',
        ],
        'drupalSettings' => [
          'ai_search_block_log' => [
            'monthly_totals' => $monthly_totals,
          ],
        ],
      ],
    ];

    // Cache for 1 hour.
    $this->cache->set($cache_key, $build, time() + 3600);

    return $build;
  }

  /**
   * Display full graphs page (legacy method for backwards compatibility).
   *
   * @return array
   *   A renderable array.
   */
  public function graphs(): array {
    // Generate cache key based on current day.
    $cache_key = 'ai_search_block_log:graphs:' . date('Y-m-d');

    // Try to get cached data.
    $cache = $this->cache->get($cache_key);
    if ($cache && !empty($cache->data)) {
      $cached_data = $cache->data;
      return [
        '#theme' => 'ai_search_block_log_graphs',
        '#chart_data' => $cached_data['chart_data'],
        '#unique_scores' => $cached_data['unique_scores'],
        '#unique_blocks' => $cached_data['unique_blocks'],
        '#user_data' => $cached_data['user_data'],
        '#block_totals' => $cached_data['block_totals'],
        '#monthly_totals' => $cached_data['monthly_totals'] ?? [],
        '#ai_analysis' => $cached_data['ai_analysis'] ?? '',
        '#scoring_stats' => $cached_data['scoring_stats'] ?? [],
        '#attached' => [
          'library' => [
            'ai_search_block_log/graphs',
          ],
          'drupalSettings' => [
            'ai_search_block_log' => [
              'chart_data' => $cached_data['chart_data'],
              'unique_scores' => $cached_data['unique_scores'],
              'unique_blocks' => $cached_data['unique_blocks'],
              'user_data' => $cached_data['user_data'],
              'block_totals' => $cached_data['block_totals'],
            ],
          ],
        ],
      ];
    }

    // No cache found, generate data.
    $connection = $this->database;

    // Get current and previous month date range.
    $current_month_start = strtotime('first day of this month 00:00:00');
    $previous_month_start = strtotime('first day of last month 00:00:00');
    $current_month_end = time();

    // Query for unique scores.
    $unique_scores_query = $connection->select('ai_search_block_log', 'l')
      ->fields('l', ['score'])
      ->distinct()
      ->condition('score', NULL, 'IS NOT NULL')
      ->orderBy('score', 'ASC');
    $unique_scores = $unique_scores_query->execute()->fetchCol();

    // Query for daily data.
    $query = $connection->select('ai_search_block_log', 'l');
    $query->addExpression('DATE(FROM_UNIXTIME(created))', 'day');
    $query->addExpression('COUNT(*)', 'total_submissions');
    $query->addExpression('AVG(score)', 'avg_score');

    // Add count for each unique score.
    foreach ($unique_scores as $score) {
      $query->addExpression(
        "SUM(CASE WHEN score = :score_$score THEN 1 ELSE 0 END)",
        "score_$score",
        [":score_$score" => $score]
      );
    }

    $query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('day')
      ->orderBy('day', 'ASC');

    $results = $query->execute()->fetchAll();

    // Create an associative array indexed by date for easy lookup.
    $data_by_date = [];
    foreach ($results as $row) {
      $data_by_date[$row->day] = $row;
    }

    // Query for block_id data (stacked line chart).
    $block_query = $connection->select('ai_search_block_log', 'l');
    $block_query->addExpression('DATE(FROM_UNIXTIME(created))', 'day');
    $block_query->fields('l', ['block_id']);
    $block_query->addExpression('COUNT(*)', 'count');
    $block_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('day')
      ->groupBy('block_id')
      ->orderBy('day', 'ASC');

    $block_results = $block_query->execute()->fetchAll();

    // Get unique block IDs.
    $unique_blocks_query = $connection->select('ai_search_block_log', 'l')
      ->fields('l', ['block_id'])
      ->distinct()
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $unique_blocks = $unique_blocks_query->execute()->fetchCol();

    // Query for unique UIDs in the timeframe.
    $unique_uids_query = $connection->select('ai_search_block_log', 'l')
      ->fields('l', ['uid'])
      ->distinct()
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $unique_uids = $unique_uids_query->execute()->fetchCol();

    // Load usernames for all UIDs.
    $user_storage = $this->entityTypeManager()->getStorage('user');
    $users = $user_storage->loadMultiple($unique_uids);
    $user_names = [];
    foreach ($users as $uid => $user) {
      $user_names[$uid] = $user->getAccountName();
    }

    // Count submissions per user.
    $user_query = $connection->select('ai_search_block_log', 'l');
    $user_query->fields('l', ['uid']);
    $user_query->addExpression('COUNT(*)', 'count');
    $user_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('uid');
    $user_results = $user_query->execute()->fetchAll();

    $user_data = [];
    foreach ($user_results as $row) {
      if ($row->uid == 0) {
        $username = 'Anonymous';
      }
      else {
        $username = $user_names[$row->uid] ?? "User {$row->uid}";
      }
      $user_data[$username] = (int) $row->count;
    }

    // Count total submissions per block_id for pie chart.
    $block_totals_query = $connection->select('ai_search_block_log', 'l');
    $block_totals_query->fields('l', ['block_id']);
    $block_totals_query->addExpression('COUNT(*)', 'count');
    $block_totals_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('block_id');
    $block_totals_results = $block_totals_query->execute()->fetchAll();

    $block_totals = [];
    foreach ($block_totals_results as $row) {
      $block_totals[$row->block_id] = (int) $row->count;
    }

    // Query for scoring statistics.
    // Total submissions in timeframe.
    $total_submissions_query = $connection->select('ai_search_block_log', 'l');
    $total_submissions_query->addExpression('COUNT(*)', 'count');
    $total_submissions_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $total_submissions = (int) $total_submissions_query->execute()->fetchField();

    // Scored submissions.
    $scored_submissions_query = $connection->select('ai_search_block_log', 'l');
    $scored_submissions_query->addExpression('COUNT(*)', 'count');
    $scored_submissions_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->isNotNull('score');
    $scored_submissions = (int) $scored_submissions_query->execute()->fetchField();

    // Find highest score in timeframe.
    $max_score_query = $connection->select('ai_search_block_log', 'l');
    $max_score_query->addExpression('MAX(score)', 'max_score');
    $max_score_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $max_score = (int) $max_score_query->execute()->fetchField();

    // Count of score=1 (poor).
    $poor_score_query = $connection->select('ai_search_block_log', 'l');
    $poor_score_query->addExpression('COUNT(*)', 'count');
    $poor_score_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->condition('score', 1, '=');
    $poor_score_count = (int) $poor_score_query->execute()->fetchField();

    // Count of highest score (good).
    $good_score_query = $connection->select('ai_search_block_log', 'l');
    $good_score_query->addExpression('COUNT(*)', 'count');
    $good_score_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->condition('score', $max_score, '=');
    $good_score_count = (int) $good_score_query->execute()->fetchField();

    // Calculate percentages.
    $scoring_stats = [
      'total_submissions' => $total_submissions,
      'scored_submissions' => $scored_submissions,
      'scoring_percentage' => $total_submissions > 0 ? round(($scored_submissions / $total_submissions) * 100, 2) : 0,
      'poor_score_count' => $poor_score_count,
      'poor_score_percentage' => $total_submissions > 0 ? round(($poor_score_count / $total_submissions) * 100, 2) : 0,
      'good_score_count' => $good_score_count,
      'good_score_percentage' => $total_submissions > 0 ? round(($good_score_count / $total_submissions) * 100, 2) : 0,
      'max_score' => $max_score,
    ];

    // Query for monthly totals (last 6 months).
    $six_months_ago = strtotime('-6 months', $current_month_start);
    $monthly_query = $connection->select('ai_search_block_log', 'l');
    $monthly_query->addExpression("DATE_FORMAT(FROM_UNIXTIME(created), '%Y-%m')", 'month');
    $monthly_query->addExpression('COUNT(*)', 'count');
    $monthly_query->condition('created', $six_months_ago, '>=')
      ->groupBy('month')
      ->orderBy('month', 'ASC');
    $monthly_results = $monthly_query->execute()->fetchAll();

    $monthly_totals = [];
    foreach ($monthly_results as $row) {
      // Format as "Nov 2024" style.
      $date = \DateTime::createFromFormat('Y-m', $row->month);
      $month_label = $date ? $date->format('M Y') : $row->month;
      $monthly_totals[$month_label] = (int) $row->count;
    }

    // Query for low-rated searches (score = 1) for AI analysis.
    $low_rated_query = $connection->select('ai_search_block_log', 'l');
    $low_rated_query->fields('l', ['question', 'response_given']);
    $low_rated_query->condition('score', 1, '=')
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->range(0, 100);
    $low_rated_results = $low_rated_query->execute()->fetchAll();

    // Generate AI analysis of low-rated searches.
    $ai_analysis = '';
    if (!empty($low_rated_results)) {
      $questions_list = '';
      foreach ($low_rated_results as $row) {
        $questions_list .= "Q: {$row->question}\nA: {$row->response_given}\n\n";
      }

      $prompt = '

INSTRUCTIONS
------------
You are summarising data for an website ai search manager. 
You write professional english. Youy don not answer a question. 
I will give you a number of questions and answers. THe people (end users ) that receives these answers were unhappy. 
From the following questions and answers: 
- find recurring themes in the questions or answers 
- explain the recurring themes very briefly. You get to write max 4 sentences. Make the sentences simple and short. Easy to understand.

RESPONSE FORMAT
-----------
You may use <i> <u> <b> <ul><li> html tags in your response.
If you find no questions or responses just say: "No data to analyse."

[QUESTIONS]
-----------
[questions]
';
      $message = str_replace('[questions]', $questions_list, $prompt);

      try {
        // Use the default chat model provider.
        $default_provider = $this->aiProviderManager
          ->getDefaultProviderForOperationType('chat');
        $model_id = $default_provider['model_id'];
        $provider_id = $default_provider['provider_id'];
        $ai_provider_model = $provider_id . '__' . $model_id;

        $provider = $this->aiProviderManager
          ->loadProviderFromSimpleOption($ai_provider_model);

        if ($provider !== NULL) {
          $temp = 0.5;
          $provider->setConfiguration(['temperature' => (float) $temp]);
          $messages[] = new ChatMessage('user', $message);
          $input = new ChatInput($messages);

          $output = $provider->chat($input, $model_id, ['ai_search_block']);
          $chatOutput = $output->getNormalized();
          $ai_analysis = $chatOutput->getText();
        }
      }
      catch (\Exception $e) {
        $ai_analysis = 'Unable to generate AI analysis: ' . $e->getMessage();
      }
    }
    else {
      $ai_analysis = 'No low-rated searches found in the selected timeframe.';
    }

    // Organize block data by date and block_id.
    $block_data_by_date = [];
    foreach ($block_results as $row) {
      if (!isset($block_data_by_date[$row->day])) {
        $block_data_by_date[$row->day] = [];
      }
      $block_data_by_date[$row->day][$row->block_id] = (int) $row->count;
    }

    // Query for average score per block_id.
    $block_avg_query = $connection->select('ai_search_block_log', 'l');
    $block_avg_query->addExpression('DATE(FROM_UNIXTIME(created))', 'day');
    $block_avg_query->fields('l', ['block_id']);
    $block_avg_query->addExpression('AVG(score)', 'avg_score');
    $block_avg_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->condition('score', NULL, 'IS NOT NULL')
      ->groupBy('day')
      ->groupBy('block_id')
      ->orderBy('day', 'ASC');

    $block_avg_results = $block_avg_query->execute()->fetchAll();

    // Organize block average score data by date and block_id.
    $block_avg_by_date = [];
    foreach ($block_avg_results as $row) {
      if (!isset($block_avg_by_date[$row->day])) {
        $block_avg_by_date[$row->day] = [];
      }
      $block_avg_by_date[$row->day][$row->block_id] = $row->avg_score ? round((float) $row->avg_score, 2) : 0;
    }

    // Generate all dates in the range.
    $all_dates = [];
    $current_date = $previous_month_start;
    while ($current_date <= $current_month_end) {
      $all_dates[] = date('Y-m-d', $current_date);
      $current_date = strtotime('+1 day', $current_date);
    }

    // Prepare data for JavaScript.
    $chart_data = [
      'labels' => [],
      'submissions' => [],
      'avg_scores' => [],
      'score_lines' => [],
      'block_lines' => [],
      'block_avg_scores' => [],
    ];

    // Initialize score lines.
    foreach ($unique_scores as $score) {
      $chart_data['score_lines'][$score] = [];
    }

    // Initialize block lines.
    foreach ($unique_blocks as $block_id) {
      $chart_data['block_lines'][$block_id] = [];
    }

    // Initialize block average score lines.
    foreach ($unique_blocks as $block_id) {
      $chart_data['block_avg_scores'][$block_id] = [];
    }

    // Fill in data for every day.
    foreach ($all_dates as $date) {
      $chart_data['labels'][] = $date;

      if (isset($data_by_date[$date])) {
        $row = $data_by_date[$date];
        $chart_data['submissions'][] = (int) $row->total_submissions;
        $chart_data['avg_scores'][] = $row->avg_score ? round((float) $row->avg_score, 2) : 0;

        foreach ($unique_scores as $score) {
          $score_field = "score_$score";
          $chart_data['score_lines'][$score][] = (int) $row->$score_field;
        }
      }
      else {
        // No data for this day, fill with zeros.
        $chart_data['submissions'][] = 0;
        $chart_data['avg_scores'][] = 0;

        foreach ($unique_scores as $score) {
          $chart_data['score_lines'][$score][] = 0;
        }
      }

      // Add block data for this day.
      foreach ($unique_blocks as $block_id) {
        if (isset($block_data_by_date[$date][$block_id])) {
          $chart_data['block_lines'][$block_id][] = $block_data_by_date[$date][$block_id];
        }
        else {
          $chart_data['block_lines'][$block_id][] = 0;
        }
      }

      // Add block average score data for this day.
      foreach ($unique_blocks as $block_id) {
        if (isset($block_avg_by_date[$date][$block_id])) {
          $chart_data['block_avg_scores'][$block_id][] = $block_avg_by_date[$date][$block_id];
        }
        else {
          $chart_data['block_avg_scores'][$block_id][] = 0;
        }
      }
    }

    // Cache the processed data for 1 hour.
    $cache_data = [
      'chart_data' => $chart_data,
      'unique_scores' => $unique_scores,
      'unique_blocks' => $unique_blocks,
      'user_data' => $user_data,
      'block_totals' => $block_totals,
      'monthly_totals' => $monthly_totals,
      'ai_analysis' => $ai_analysis,
      'scoring_stats' => $scoring_stats,
    ];
    $this->cache->set($cache_key, $cache_data, time() + 3600);

    $build = [
      '#theme' => 'ai_search_block_log_graphs',
      '#chart_data' => $chart_data,
      '#unique_scores' => $unique_scores,
      '#unique_blocks' => $unique_blocks,
      '#user_data' => $user_data,
      '#block_totals' => $block_totals,
      '#monthly_totals' => $monthly_totals,
      '#ai_analysis' => $ai_analysis,
      '#scoring_stats' => $scoring_stats,
      '#attached' => [
        'library' => [
          'ai_search_block_log/graphs',
        ],
        'drupalSettings' => [
          'ai_search_block_log' => [
            'chart_data' => $chart_data,
            'unique_scores' => $unique_scores,
            'unique_blocks' => $unique_blocks,
            'user_data' => $user_data,
            'block_totals' => $block_totals,
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * Display AI Analysis page.
   *
   * @return array
   *   A renderable array.
   */
  public function graphsAiAnalysis(): array {
    // Generate cache key based on current day.
    $cache_key = 'ai_search_block_log:graphs_ai_analysis:' . date('Y-m-d');

    // Try to get cached data.
    $cache = $this->cache->get($cache_key);
    if ($cache && !empty($cache->data)) {
      return $cache->data;
    }

    $connection = $this->database;

    $current_month_start = strtotime('first day of this month 00:00:00');
    $previous_month_start = strtotime('first day of last month 00:00:00');
    $current_month_end = time();

    // Get AI analysis with feedback.
    $ai_analysis = $this->getAiAnalysisWithFeedback($connection, $previous_month_start, $current_month_end);

    $build = [
      '#theme' => 'ai_search_block_log_graphs_ai_analysis',
      '#ai_analysis' => $ai_analysis,
    ];

    // Cache for 1 hour.
    $this->cache->set($cache_key, $build, time() + 3600);

    return $build;
  }

  /**
   * Display Search Stats page.
   *
   * @return array
   *   A renderable array.
   */
  public function graphsSearchStats(): array {
    // Generate cache key based on current day.
    $cache_key = 'ai_search_block_log:graphs_search_stats:' . date('Y-m-d');

    // Try to get cached data.
    $cache = $this->cache->get($cache_key);
    if ($cache && !empty($cache->data)) {
      return $cache->data;
    }

    $connection = $this->database;

    $current_month_start = strtotime('first day of this month 00:00:00');
    $previous_month_start = strtotime('first day of last month 00:00:00');
    $current_month_end = time();

    $data = $this->getSearchStatsData($connection, $previous_month_start, $current_month_end);

    $build = [
      '#theme' => 'ai_search_block_log_graphs_search_stats',
      '#chart_data' => $data['chart_data'],
      '#unique_blocks' => $data['unique_blocks'],
      '#user_data' => $data['user_data'],
      '#block_totals' => $data['block_totals'],
      '#attached' => [
        'library' => [
          'ai_search_block_log/graphs',
        ],
        'drupalSettings' => [
          'ai_search_block_log' => [
            'chart_data' => $data['chart_data'],
            'unique_blocks' => $data['unique_blocks'],
            'user_data' => $data['user_data'],
            'block_totals' => $data['block_totals'],
          ],
        ],
      ],
    ];

    // Cache for 1 hour.
    $this->cache->set($cache_key, $build, time() + 3600);

    return $build;
  }

  /**
   * Display Scoring Stats page.
   *
   * @return array
   *   A renderable array.
   */
  public function graphsScoringStats(): array {
    // Generate cache key based on current day.
    $cache_key = 'ai_search_block_log:graphs_scoring_stats:' . date('Y-m-d');

    // Try to get cached data.
    $cache = $this->cache->get($cache_key);
    if ($cache && !empty($cache->data)) {
      return $cache->data;
    }

    $connection = $this->database;

    $current_month_start = strtotime('first day of this month 00:00:00');
    $previous_month_start = strtotime('first day of last month 00:00:00');
    $current_month_end = time();

    $data = $this->getScoringStatsData($connection, $previous_month_start, $current_month_end);

    $build = [
      '#theme' => 'ai_search_block_log_graphs_scoring_stats',
      '#chart_data' => $data['chart_data'],
      '#unique_scores' => $data['unique_scores'],
      '#attached' => [
        'library' => [
          'ai_search_block_log/graphs',
        ],
        'drupalSettings' => [
          'ai_search_block_log' => [
            'chart_data' => $data['chart_data'],
            'unique_scores' => $data['unique_scores'],
          ],
        ],
      ],
    ];

    // Cache for 1 hour.
    $this->cache->set($cache_key, $build, time() + 3600);

    return $build;
  }

  /**
   * Helper: Get scoring statistics.
   */
  private function getScoringStats($connection, $previous_month_start, $current_month_end): array {
    // Total submissions in timeframe.
    $total_submissions_query = $connection->select('ai_search_block_log', 'l');
    $total_submissions_query->addExpression('COUNT(*)', 'count');
    $total_submissions_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $total_submissions = (int) $total_submissions_query->execute()->fetchField();

    // Scored submissions.
    $scored_submissions_query = $connection->select('ai_search_block_log', 'l');
    $scored_submissions_query->addExpression('COUNT(*)', 'count');
    $scored_submissions_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->isNotNull('score');
    $scored_submissions = (int) $scored_submissions_query->execute()->fetchField();

    // Find highest score in timeframe.
    $max_score_query = $connection->select('ai_search_block_log', 'l');
    $max_score_query->addExpression('MAX(score)', 'max_score');
    $max_score_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $max_score = (int) $max_score_query->execute()->fetchField();

    // Count of score=1 (poor).
    $poor_score_query = $connection->select('ai_search_block_log', 'l');
    $poor_score_query->addExpression('COUNT(*)', 'count');
    $poor_score_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->condition('score', 1, '=');
    $poor_score_count = (int) $poor_score_query->execute()->fetchField();

    // Count of highest score (good).
    $good_score_query = $connection->select('ai_search_block_log', 'l');
    $good_score_query->addExpression('COUNT(*)', 'count');
    $good_score_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->condition('score', $max_score, '=');
    $good_score_count = (int) $good_score_query->execute()->fetchField();

    return [
      'total_submissions' => $total_submissions,
      'scored_submissions' => $scored_submissions,
      'scoring_percentage' => $total_submissions > 0 ? round(($scored_submissions / $total_submissions) * 100, 2) : 0,
      'poor_score_count' => $poor_score_count,
      'poor_score_percentage' => $total_submissions > 0 ? round(($poor_score_count / $total_submissions) * 100, 2) : 0,
      'good_score_count' => $good_score_count,
      'good_score_percentage' => $total_submissions > 0 ? round(($good_score_count / $total_submissions) * 100, 2) : 0,
      'max_score' => $max_score,
    ];
  }

  /**
   * Helper: Get monthly totals for last 6 months.
   */
  private function getMonthlyTotals($connection, $current_month_start): array {
    $six_months_ago = strtotime('-6 months', $current_month_start);
    $monthly_query = $connection->select('ai_search_block_log', 'l');
    $monthly_query->addExpression("DATE_FORMAT(FROM_UNIXTIME(created), '%Y-%m')", 'month');
    $monthly_query->addExpression('COUNT(*)', 'count');
    $monthly_query->condition('created', $six_months_ago, '>=')
      ->groupBy('month')
      ->orderBy('month', 'ASC');
    $monthly_results = $monthly_query->execute()->fetchAll();

    $monthly_totals = [];
    foreach ($monthly_results as $row) {
      $date = \DateTime::createFromFormat('Y-m', $row->month);
      $month_label = $date ? $date->format('M Y') : $row->month;
      $monthly_totals[$month_label] = (int) $row->count;
    }

    return $monthly_totals;
  }

  /**
   * Helper: Get AI analysis with feedback data.
   */
  private function getAiAnalysisWithFeedback($connection, $previous_month_start, $current_month_end): string {
    // First, try to get low-rated searches (score <= 2 or with feedback).
    $low_rated_query = $connection->select('ai_search_block_log', 'l');
    $low_rated_query->fields('l', ['question', 'response_given', 'feedback', 'score']);

    // Create OR condition group for low scores or feedback.
    $or = $low_rated_query->orConditionGroup()
      ->condition('score', 2, '<=')
      ->isNotNull('feedback');

    $low_rated_query->condition($or)
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->orderBy('created', 'DESC')
      ->range(0, 100);
    $low_rated_results = $low_rated_query->execute()->fetchAll();

    if (empty($low_rated_results)) {
      return 'No low-rated searches or feedback found in the selected timeframe.';
    }

    $questions_list = '';

    foreach ($low_rated_results as $row) {
      $score_text = !empty($row->score) ? " [Score: {$row->score}]" : " [Not scored]";
      $questions_list .= "Q: {$row->question}{$score_text}\nA: {$row->response_given}\n";
      if (!empty($row->feedback)) {
        $questions_list .= "Feedback: {$row->feedback}\n";
      }
      $questions_list .= "\n";
    }

    // Get the configured prompt or use default.
    $config = \Drupal::config('ai_search_block_log.settings');
    $prompt = $config->get('ai_analysis_prompt');

    // If no configured prompt, use default.
    if (empty($prompt)) {
      $prompt = 'INSTRUCTIONS
------------
You are summarising data for a website ai search manager.
You write professional english. You do not answer a question.
I will give you a number of questions, answers with scores, and user feedback. These are searches that received low scores (1-2) or had user feedback provided.
From the following questions, answers, and feedback:
- find recurring themes in the questions, answers, or feedback
- identify common problems or issues
- explain the recurring themes very briefly. You get to write max 5 sentences. Make the sentences simple and short. Easy to understand.

RESPONSE FORMAT
-----------
You may use <i> <u> <b> <ul><li> html tags in your response.
If you find no questions or responses just say: "No data to analyse."

[QUESTIONS AND ANSWERS WITH SCORES]
-----------
[questions]';
    }

    $message = str_replace('[questions]', $questions_list, $prompt);

    try {
      $default_provider = $this->aiProviderManager->getDefaultProviderForOperationType('chat');
      $model_id = $default_provider['model_id'];
      $provider_id = $default_provider['provider_id'];
      $ai_provider_model = $provider_id . '__' . $model_id;

      $provider = $this->aiProviderManager->loadProviderFromSimpleOption($ai_provider_model);

      if ($provider !== NULL) {
        $temp = 0.5;
        $provider->setConfiguration(['temperature' => (float) $temp]);
        $messages[] = new ChatMessage('user', $message);
        $input = new ChatInput($messages);

        $output = $provider->chat($input, $model_id, ['ai_search_block']);
        $chatOutput = $output->getNormalized();
        return $chatOutput->getText();
      }
    }
    catch (\Exception $e) {
      return 'Unable to generate AI analysis: ' . $e->getMessage();
    }

    return 'Unable to generate AI analysis.';
  }

  /**
   * Helper: Get search stats data.
   */
  private function getSearchStatsData($connection, $previous_month_start, $current_month_end): array {
    // Query for daily submissions.
    $query = $connection->select('ai_search_block_log', 'l');
    $query->addExpression('DATE(FROM_UNIXTIME(created))', 'day');
    $query->addExpression('COUNT(*)', 'total_submissions');
    $query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('day')
      ->orderBy('day', 'ASC');
    $results = $query->execute()->fetchAll();

    $data_by_date = [];
    foreach ($results as $row) {
      $data_by_date[$row->day] = $row;
    }

    // Query for block_id data.
    $block_query = $connection->select('ai_search_block_log', 'l');
    $block_query->addExpression('DATE(FROM_UNIXTIME(created))', 'day');
    $block_query->fields('l', ['block_id']);
    $block_query->addExpression('COUNT(*)', 'count');
    $block_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('day')
      ->groupBy('block_id')
      ->orderBy('day', 'ASC');
    $block_results = $block_query->execute()->fetchAll();

    // Get unique block IDs.
    $unique_blocks_query = $connection->select('ai_search_block_log', 'l')
      ->fields('l', ['block_id'])
      ->distinct()
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $unique_blocks = $unique_blocks_query->execute()->fetchCol();

    // Get user data.
    $unique_uids_query = $connection->select('ai_search_block_log', 'l')
      ->fields('l', ['uid'])
      ->distinct()
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=');
    $unique_uids = $unique_uids_query->execute()->fetchCol();

    $user_storage = $this->entityTypeManager()->getStorage('user');
    $users = $user_storage->loadMultiple($unique_uids);
    $user_names = [];
    foreach ($users as $uid => $user) {
      $user_names[$uid] = $user->getAccountName();
    }

    $user_query = $connection->select('ai_search_block_log', 'l');
    $user_query->fields('l', ['uid']);
    $user_query->addExpression('COUNT(*)', 'count');
    $user_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('uid');
    $user_results = $user_query->execute()->fetchAll();

    $user_data = [];
    foreach ($user_results as $row) {
      if ($row->uid == 0) {
        $username = 'Anonymous';
      }
      else {
        $username = $user_names[$row->uid] ?? "User {$row->uid}";
      }
      $user_data[$username] = (int) $row->count;
    }

    // Count total submissions per block_id for pie chart.
    $block_totals_query = $connection->select('ai_search_block_log', 'l');
    $block_totals_query->fields('l', ['block_id']);
    $block_totals_query->addExpression('COUNT(*)', 'count');
    $block_totals_query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('block_id');
    $block_totals_results = $block_totals_query->execute()->fetchAll();

    $block_totals = [];
    foreach ($block_totals_results as $row) {
      $block_totals[$row->block_id] = (int) $row->count;
    }

    // Organize block data by date.
    $block_data_by_date = [];
    foreach ($block_results as $row) {
      if (!isset($block_data_by_date[$row->day])) {
        $block_data_by_date[$row->day] = [];
      }
      $block_data_by_date[$row->day][$row->block_id] = (int) $row->count;
    }

    // Generate all dates in range.
    $all_dates = [];
    $current_date = $previous_month_start;
    while ($current_date <= $current_month_end) {
      $all_dates[] = date('Y-m-d', $current_date);
      $current_date = strtotime('+1 day', $current_date);
    }

    // Prepare chart data.
    $chart_data = [
      'labels' => [],
      'submissions' => [],
      'block_lines' => [],
    ];

    // Initialize block lines.
    foreach ($unique_blocks as $block_id) {
      $chart_data['block_lines'][$block_id] = [];
    }

    foreach ($all_dates as $date) {
      $chart_data['labels'][] = $date;

      if (isset($data_by_date[$date])) {
        $row = $data_by_date[$date];
        $chart_data['submissions'][] = (int) $row->total_submissions;
      }
      else {
        $chart_data['submissions'][] = 0;
      }

      foreach ($unique_blocks as $block_id) {
        if (isset($block_data_by_date[$date][$block_id])) {
          $chart_data['block_lines'][$block_id][] = $block_data_by_date[$date][$block_id];
        }
        else {
          $chart_data['block_lines'][$block_id][] = 0;
        }
      }
    }

    return [
      'chart_data' => $chart_data,
      'unique_blocks' => $unique_blocks,
      'user_data' => $user_data,
      'block_totals' => $block_totals,
    ];
  }

  /**
   * Helper: Get scoring stats data.
   */
  private function getScoringStatsData($connection, $previous_month_start, $current_month_end): array {
    // Query for unique scores in timeframe.
    $unique_scores_query = $connection->select('ai_search_block_log', 'l')
      ->fields('l', ['score'])
      ->distinct()
      ->condition('score', NULL, 'IS NOT NULL')
      ->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->orderBy('score', 'ASC');
    $unique_scores = $unique_scores_query->execute()->fetchCol();

    // Query for daily data.
    $query = $connection->select('ai_search_block_log', 'l');
    $query->addExpression('DATE(FROM_UNIXTIME(created))', 'day');
    $query->addExpression('AVG(score)', 'avg_score');

    foreach ($unique_scores as $score) {
      $query->addExpression(
        "SUM(CASE WHEN score = :score_$score THEN 1 ELSE 0 END)",
        "score_$score",
        [":score_$score" => $score]
      );
    }

    $query->condition('created', $previous_month_start, '>=')
      ->condition('created', $current_month_end, '<=')
      ->groupBy('day')
      ->orderBy('day', 'ASC');

    $results = $query->execute()->fetchAll();

    $data_by_date = [];
    foreach ($results as $row) {
      $data_by_date[$row->day] = $row;
    }

    // Generate all dates in range.
    $all_dates = [];
    $current_date = $previous_month_start;
    while ($current_date <= $current_month_end) {
      $all_dates[] = date('Y-m-d', $current_date);
      $current_date = strtotime('+1 day', $current_date);
    }

    // Prepare chart data.
    $chart_data = [
      'labels' => [],
      'avg_scores' => [],
      'score_lines' => [],
    ];

    // Initialize score lines.
    foreach ($unique_scores as $score) {
      $chart_data['score_lines'][$score] = [];
    }

    foreach ($all_dates as $date) {
      $chart_data['labels'][] = $date;

      if (isset($data_by_date[$date])) {
        $row = $data_by_date[$date];
        $chart_data['avg_scores'][] = $row->avg_score ? round((float) $row->avg_score, 2) : 0;

        foreach ($unique_scores as $score) {
          $score_field = "score_$score";
          $chart_data['score_lines'][$score][] = (int) $row->$score_field;
        }
      }
      else {
        $chart_data['avg_scores'][] = 0;

        foreach ($unique_scores as $score) {
          $chart_data['score_lines'][$score][] = 0;
        }
      }
    }

    return [
      'chart_data' => $chart_data,
      'unique_scores' => $unique_scores,
    ];
  }

}
