<?php

declare(strict_types=1);

namespace Drupal\ai_search_block;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\StreamedChatMessageIteratorInterface;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\search_api\Query\ResultSetInterface;
use League\HTMLToMarkdown\Converter\TableConverter;
use League\HTMLToMarkdown\HtmlConverter;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * The Helper service to do RAG stuff.
 */
class AiSearchBlockHelper implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The configuration parameters passed in.
   *
   * @var array<string, mixed>
   */
  private array $configuration;

  /**
   * The converter.
   *
   * @var \League\HTMLToMarkdown\HtmlConverter
   */
  private HtmlConverter $converter;

  /**
   * The id of the log row.
   *
   * @var int
   */
  public int $logId;

  /**
   * Constructs an AiSearchBlockHelper instance.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $tmpStore
   *   The temp store.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\ai\AiProviderPluginManager $aiProviderManager
   *   The AI provider manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected PrivateTempStoreFactory $tmpStore,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RendererInterface $renderer,
    protected AiProviderPluginManager $aiProviderManager,
    protected RequestStack $requestStack,
    protected LanguageManagerInterface $languageManager,
    protected AccountProxyInterface $currentUser,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
  ) {
    $this->converter = new HtmlConverter();
    $this->converter->getConfig()->setOption('strip_tags', TRUE);
    $this->converter->getConfig()->setOption('strip_placeholder_links', TRUE);
    $this->converter->getConfig()->setOption('header_style', 'atx');
    $this->converter->getEnvironment()->addConverter(new TableConverter());
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $container->get('tempstore.private'),
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('ai.provider'),
      $container->get('request_stack'),
      $container->get('language_manager'),
      $container->get('current_user'),
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * Set the config for this Search.
   *
   * @param array<string, mixed> $config
   *   The array with configuration.
   */
  public function setConfig(array $config): void {
    $this->configuration = $config;
  }

  /**
   * The block ID (for logging).
   *
   * @param string $block_id
   *   The id of the block.
   */
  public function setBlockId(string $block_id): void {
    // Block ID is no longer stored as a property.
  }

  /**
   * Test if valid input.
   *
   * @param string $query
   *   The question.
   *
   * @return bool
   *   If the question is valid or not.
   */
  private function validInput(string $query): bool {
    if ($this->configuration['block_enabled'] === 1) {
      $lines = explode(PHP_EOL, $this->configuration['block_words']);
      foreach ($lines as $line) {
        $line = trim($line);
        if (str_contains($query, $line)) {
          // Not valid this FALSE.
          return FALSE;
        }
      }
    }

    return TRUE;
  }

  /**
   * Take rag action.
   *
   * @param string $query
   *   The question from the user.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\StreamedResponse|string
   *   The streamed response.
   *
   * @throws \Exception
   */
  public function searchRagAction(string $query): JsonResponse|StreamedResponse|string {
    if (!$this->validInput($query)) {
      return $this->giveMeAnError($this->configuration['block_response']);
    }

    if (!empty($this->configuration['database'])) {
      $rag_database = $this->configuration;
    }
    if (!isset($rag_database)) {
      return $this->giveMeAnError('[ERROR] No RAG database found.');
    }
    $results = $this->getRagResults($rag_database, $query);
    $min_results = $this->configuration['min_results'];
    if ($results->getResultCount() < $min_results) {
      return $this->giveMeAnError($this->configuration['no_results_message']);
    }

    return $this->renderRagResponseAsString($results, $query, $rag_database);
  }

  /**
   * Returns the errors.
   *
   * @param string $msg
   *   The message for the error.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   The Json response.
   */
  public function giveMeAnError(string $msg): StreamedResponse {
    $msg = '<p class="error">' . $msg . '</p>';
    $parts = str_split($msg, 4);

    return $this->streamBackResponse($parts, 'string', '', []);
  }

  /**
   * Full entity check with a LLM checking the rendered entity.
   *
   * @param \Drupal\search_api\Item\ItemInterface[] $result_items
   *   The result to check.
   * @param string $query_string
   *   The query to search for.
   * @param array $rag_database
   *   The RAG database array data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\StreamedResponse|string
   *   The response.
   */
  protected function fullEntityCheck(
    array $result_items,
    string $query_string,
    array $rag_database,
  ): JsonResponse|StreamedResponse|string {
    $entity_list = [];
    $rendered_entities = [];
    if ($this->configuration['render_mode'] === 'chunks') {
      foreach ($result_items as $result) {
        $chunk = $result->getExtraData('content');
        $chunk = $this->cleanupMarkdown($chunk, NULL);
        $rendered_entities[] = $chunk;
      }
    }

    if ($this->configuration['render_mode'] === 'node') {
      foreach ($result_items as $result) {
        $entity_string = $result->getId();
        // Load the entity from search api key.
        // @todo probably exists a function for this.
        $parts = explode(':', $entity_string);
        $entity_parts = $parts[1] ?? '';
        $lang = $parts[2] ?? '';
        $entity_info = explode('/', $entity_parts);
        $entity_type = $entity_info[0] ?? '';
        $entity_id = $entity_info[1] ?? '';
        /** @var \Drupal\Core\Entity\ContentEntityBase $entity */
        $entity = $this->entityTypeManager->getStorage($entity_type)
          ->load($entity_id);
        $entity_list[$entity_id] = [
          'lang' => $lang,
          'entity' => $entity,
          'entity_type' => $entity_type,
        ];
      }

      // $entities are filtered now.
      foreach ($entity_list as $entity_id => $entity_array) {
        $lang = $entity_array['lang'];
        $entity = $entity_array['entity'];
        $entity_type = $entity_array['entity_type'];
        // Get translated if possible.
        if (
          $entity->hasTranslation($lang)
          && $entity->language()->getId() !== $lang
        ) {
          $entity = $entity->getTranslation($lang);
        }
        // Render the entity in selected view mode.
        $view_mode = $this->configuration['rendered_view_mode'] ?? 'full';
        $view_builder = $this->entityTypeManager
          ->getViewBuilder($entity_type);
        $pre_render_entity = $view_builder->view($entity, $view_mode);
        $rendered = $this->renderer->render($pre_render_entity);
        $rendered = $this->cleanupHtml($rendered, $entity);
        $markdown = $this->converter->convert((string) $rendered);
        $markdown = $this->cleanupMarkdown($markdown, $entity);

        // Get the node URL.
        $node_url = $entity->toUrl('canonical', ['absolute' => TRUE])->toString();

        // Format the node content with header and footer.
        $formatted_node = "\n>>>>>> BEGIN ENTITY {$entity_id} <<<<<<<<\n";
        $formatted_node .= "ENTITY_URL:  {$node_url}\n\n";
        $formatted_node .= "ENTITY CONTENT:\n\n";
        $formatted_node .= "{$markdown}\n\n";
        $formatted_node .= ">>>>>> END ENTITY {$entity_id} <<<<<<<<";

        $rendered_entities[] = $formatted_node;
      }
    }

    $message = str_replace([
      '[question]',
      '[entity]',
    ], [
      $query_string,
      implode("\n\n", $rendered_entities),
    ], $this->configuration['aggregated_llm']);

    foreach ($this->getPrePromptDrupalContext() as $key => $replace) {
      $message = str_replace('[' . $key . ']', (string) $replace, $message);
    }

    $tomorrow = strtotime('+ 1 day');
    $yesterday = strtotime('- 1 day');
    $date_today = date('l M j G:i:s T Y');
    $date_tomorrow = date('l M j G:i:s T Y', $tomorrow);
    $date_yesterday = date('l M j G:i:s T Y', $yesterday);
    $time_now = date('H:i:s');

    $message = str_replace([
      '[time_now]',
      '[date_today]',
      '[date_tomorrow]',
      '[date_yesterday]',
    ], [$time_now, $date_today, $date_tomorrow, $date_yesterday], $message);

    // Now we have the entity, we can check it with the LLM.
    $ai_provider_model = $this->configuration['llm_model'];

    if ($ai_provider_model === '') {
      $default_provider = $this->aiProviderManager
        ->getDefaultProviderForOperationType('chat');
      $model_id = $default_provider['model_id'];
      $provider_id = $default_provider['provider_id'];
      $ai_provider_model = $provider_id . '__' . $model_id;
      $ai_model_to_use = $model_id;
    }
    else {
      $parts = explode('__', $ai_provider_model);
      $ai_model_to_use = $parts[1];
    }
    $provider = $this->aiProviderManager
      ->loadProviderFromSimpleOption($ai_provider_model);
    if ($provider === NULL) {
      $error_msg = 'Provider ' . $ai_provider_model . ' does not exist.';
      throw new \Exception($error_msg);
    }

    $temp = $this->configuration['llm_temp'] ?? 0.5;
    $provider->setConfiguration(['temperature' => (float) $temp]);
    $this->moduleHandler->alter('ai_search_block_prompt', $message);

    // Check if we need to split the message into system and user parts.
    $messages = [];
    if (str_contains($message, '-----! SPLIT !-----')) {
      $split_parts = explode('---! SPLIT !---', $message, 2);
      $message_system = $split_parts[0];
      $message_user = $split_parts[1];
      $messages[] = new ChatMessage('system', trim($message_system));
      $messages[] = new ChatMessage('user', trim($message_user));
    }
    else {
      $messages[] = new ChatMessage('user', $message);
    }

    $input = new ChatInput($messages);

    if ($this->configuration['stream']) {
      $input->setStreamedOutput(TRUE);
      $output = $provider->chat($input, $ai_model_to_use, [
        'ai_search_block',
      ]);
      $response = $output->getNormalized();
      if ($response instanceof StreamedChatMessageIteratorInterface) {
        $entity_ids = array_keys($entity_list);
        return $this->streamBackResponse(
          $response,
          'message',
          $message,
          $entity_ids
        );
      }

      // Ai models that don't stream back?
      $output = $provider->chat($input, $ai_model_to_use, ['ai_search_block']);
      /** @var \Drupal\ai\OperationType\Chat\ChatMessage $chatOutput */
      $chatOutput = $output->getNormalized();
      $response = $chatOutput->getText() . "\n";
      $entity_ids = array_keys($entity_list);
      $this->logResponse($response, $message, $entity_ids);

      return $response;
    }

    $output = $provider->chat($input, $ai_model_to_use, ['ai_search_block']);
    /** @var \Drupal\ai\OperationType\Chat\ChatMessage $chatOutput */
    $chatOutput = $output->getNormalized();
    $response = $chatOutput->getText() . "\n";
    $entity_ids = array_keys($entity_list);
    $this->logResponse($response, $message, $entity_ids);

    return new JsonResponse([
      'response' => $response,
      'log_id' => $this->logId,
    ]);
  }

  /**
   * Log the response to the log.
   *
   * @param string $response
   *   The actual response.
   * @param string $prompt
   *   The prompt used for the LLM.
   * @param array $items
   *   The items used to generate a response.
   */
  private function logResponse(
    string $response,
    string $prompt,
    array $items,
  ): void {
    if ($this->moduleHandler->moduleExists('ai_search_block_log')) {
      ai_search_block_log_update($this->logId, [
        'prompt_used' => $prompt,
        'response_given' => $response,
        'detailed_output' => Json::encode($items),
      ]);
    }
  }

  /**
   * Clean up the HTML of the rendered entity.
   *
   * @param \Drupal\Component\Render\MarkupInterface|string $html
   *   The HTML.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity (context).
   *
   * @return \Drupal\Component\Render\MarkupInterface|string
   *   The cleaned up html.
   */
  private function cleanupHtml(
    MarkupInterface|string $html,
    EntityInterface $entity,
  ): MarkupInterface|string {
    $this->moduleHandler->alter('ai_search_block_entity_html', $html, $entity);
    return $html;
  }

  /**
   * Clean up the markdown of the entity.
   *
   * @param string $markdown
   *   The markdown that will end up in the prompt.
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The context entity.
   *
   * @return string
   *   The cleaned markdown.
   */
  private function cleanupMarkdown(
    string $markdown,
    ?EntityInterface $entity,
  ): string {
    // First cleanup multiple empty lines.
    $lines = explode(PHP_EOL, $markdown);
    $newlines = [];
    $prev = NULL;
    foreach ($lines as $line) {
      $newline = trim($line, "\t");
      if ($prev === $newline && $newline === '') {
        continue;
        // Implicit cleanup of duplicate empty lines.
      }
      $newlines[] = $newline;
      $prev = $newline;
    }
    $markdown = implode(PHP_EOL, $newlines);
    $this->moduleHandler->alter(
      'ai_search_block_entity_markdown',
      $markdown,
      $entity
    );
    return $markdown;
  }

  /**
   * Get preprompt Drupal context.
   *
   * @return array
   *   This is the Drupal context that you can add to the pre prompt.
   */
  public function getPrePromptDrupalContext(): array {
    $context = [];
    $current_request = $this->requestStack->getCurrentRequest();

    $is_authenticated = $this->currentUser->isAuthenticated();
    $login_status = $is_authenticated ? 'is logged in' : 'is not logged in';
    $context['is_logged_in'] = $login_status;
    $context['user_roles'] = implode(', ', $this->currentUser->getRoles());
    $context['user_id'] = $this->currentUser->id();
    $context['user_name'] = $this->currentUser->getDisplayName();
    $context['user_language'] = $this->currentUser->getPreferredLangcode();
    $context['user_timezone'] = $this->currentUser->getTimeZone();
    $context['page_path'] = $current_request?->getRequestUri();
    $context['page_language'] = $this->languageManager->getCurrentLanguage()
      ->getId();
    $context['site_name'] = $this->configFactory->get('system.site')
      ->get('name');

    return $context;
  }

  /**
   * Process RAG.
   *
   * @param array $rag_database
   *   The RAG database array data.
   * @param string $query_string
   *   The query to search for (optional).
   *
   * @return \Drupal\search_api\Query\ResultSetInterface
   *   The RAG response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getRagResults(
    array $rag_database,
    string $query_string,
  ): ResultSetInterface {
    $rag_storage = $this->entityTypeManager->getStorage('search_api_index');
    /** @var \Drupal\search_api\Entity\Index|null $index */
    $index = $rag_storage->load($rag_database['database']);
    if (!$index) {
      throw new \Exception('RAG database not found.');
    }

    try {
      $query = $index->query([
        'limit' => $this->configuration['max_results'],
      ]);
      $query->setOption('search_api_bypass_access', TRUE);
      $query->setOption('search_api_ai_get_chunks_result', 'rendered');

      // Apply the prefix template to the query string if enabled.
      $queries = $this->applyQueryPrefix($query_string);

      $query->keys($queries);
      $results = $query->execute();
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to search: ' . $e->getMessage());
    }

    return $results;
  }

  /**
   * Applies the prefix template to the query string if enabled.
   *
   * @param string $query_string
   *   The original query string.
   *
   * @return string
   *   The potentially modified query string with prefix applied.
   */
  protected function applyQueryPrefix(string $query_string): string {
    $enable_prefix = !empty($this->configuration['enable_prefix']);
    $has_template = !empty($this->configuration['prefix_template']);
    if ($enable_prefix && $has_template) {
      $prefix_template = $this->configuration['prefix_template'];
      // Replace {query} placeholder with the actual query string.
      return str_replace('{query}', $query_string, $prefix_template);
    }

    return $query_string;
  }

  /**
   * Render the RAG response as string.
   *
   * @param \Drupal\search_api\Query\ResultSetInterface $results
   *   The RAG results.
   * @param string $query
   *   The query to search for (optional).
   * @param array $rag_database
   *   The RAG database array data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\StreamedResponse|string
   *   The RAG response.
   */
  protected function renderRagResponseAsString(
    ResultSetInterface $results,
    string $query,
    array $rag_database,
  ): JsonResponse|StreamedResponse|string {
    $result_items = [];
    foreach ($results->getResultItems() as $result) {
      $score_threshold = (float) $this->configuration['score_threshold'];
      if ($score_threshold > $result->getScore()) {
        continue;
      }

      // Resolve the entity behind the hit.
      // Works for both 'chunks' and 'node' modes as long as
      // drupal_entity_id is present in extra data.
      $entity_string = (string) $result->getExtraData('drupal_entity_id');
      if ($entity_string === '') {
        // Do not surface hits that cannot be mapped back to an entity.
        continue;
      }
      // Parse "entity:TYPE/ID[:LANG]" safely.
      $pattern = '/^entity:([a-z0-9_]+)\/(\d+)(?::([a-z0-9_\-]+))?$/i';
      if (!preg_match($pattern, $entity_string, $m)) {
        continue;
      }
      [$entity_type, $entity_id] = [$m[1], (int) $m[2]];
      if (!$this->entityTypeManager->hasDefinition($entity_type)) {
        continue;
      }
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $entity = $storage->load($entity_id);
      // Only include if the current user may view it.
      if (!$entity || !$entity->access('view', $this->currentUser)) {
        continue;
      }
      // Optional belt-and-suspenders for nodes:
      $is_node = $entity_type === 'node';
      $has_method = method_exists($entity, 'isPublished');
      $is_unpublished = $has_method && !$entity->isPublished();
      if ($is_node && $is_unpublished) {
        continue;
      }
      $result_items[] = $result;
    }

    if (!empty($result_items)) {
      return $this->fullEntityCheck($result_items, $query, $rag_database);
    }

    if ($this->configuration['stream']) {
      $parts = str_split($this->configuration['no_results_message'], 4);
      return $this->streamBackResponse($parts, 'string', '', $result_items);
    }
    else {
      if ($this->moduleHandler->moduleExists('ai_search_block_log')) {
        ai_search_block_log_update($this->logId, [
          'response_given' => $this->configuration['no_results_message'],
          'detailed_output' => Json::encode($results),
        ]);
      }
      return new JsonResponse([
        'response' => $this->configuration['no_results_message'],
        'log_id' => $this->logId,
      ]);
    }
  }

  /**
   * Streams back the response so it comes to the frontend nice and fluid.
   *
   * @param array|\Drupal\ai\OperationType\Chat\StreamedChatMessageIteratorInterface $parts
   *   The parts of the response (stream).
   * @param string $type
   *   The type (is it a string or a message).
   * @param string $prompt
   *   The actual prompt to the LLM.
   * @param array $result_items
   *   The items used to create the response.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   The stream with the response.
   */
  private function streamBackResponse($parts, $type, $prompt, $result_items) {
    return new StreamedResponse(function () use ($type, $parts, $prompt, $result_items) {
      $log_output = '';
      foreach ($parts as $part) {
        $item = [];
        $item['in_html'] = FALSE;
        $item['log_id'] = $this->logId;
        if ($type == 'string') {
          $item['answer_piece'] = $part;
        }
        else {
          $item['answer_piece'] = $part->getText();
        }
        $out = Json::encode($item);
        $log_output .= $item['answer_piece'];
        unset($item);
        echo $out . '|§|';
        ob_flush();
        flush();
        if ($type == 'string') {
          usleep(50000);
        }
      }
      $this->logResponse($log_output, $prompt, $result_items);
    }, 200, [
      'Cache-Control' => 'no-cache, must-revalidate',
      'Content-Type' => 'text/event-stream',
      'X-Accel-Buffering' => 'no',
    ]);
  }

}
