(function ($, Drupal, drupalSettings, once) {
  
  // Utility to dispatch custom status events
  function dispatchStatusEvent(status, extra) {
    const event = new CustomEvent('ai-search-status', {
      detail: { status, ...(extra || {}) },
    });
    document.dispatchEvent(event);
  }

  // Centralized logging function for AI Search Block
  function aiSearchBlockConsoleLog(...args) {
    // Enable/disable logging by checking settings or setting a flag
    const enableLogging = drupalSettings.ai_search_block?.enable_debug_logging === true;
    if (enableLogging && typeof console !== 'undefined' && console.log) {
      console.log('[AI Search Block]', ...args);
    }
  }

  Drupal.behaviors.aiSearchBlock = {

    attach(context, settings) {
      Drupal.behaviors.aiSearchBlock.clipboard(context, settings);
      Drupal.behaviors.aiSearchBlock.feedback(context, settings);
      Drupal.behaviors.aiSearchBlock.animatePlaceholder(context, settings);

      // Attach the submit handler only once per form.
      once('aiSearchForm', '.ai-search-block-form', context).forEach(
        function (formElem) {
          const $form = $(formElem);

          // Locate the results output region and the suffix text.
          // Prefer the per-block response target set on the submit
          // button (`data-ai-ajax`). This ensures we use the response
          // block that matches this form instance when multiple blocks
          // are present on the page.
          let $resultsBlock = $();
          let $suffixText = $();
          const $aiAjax = $form.find('[data-ai-ajax]').first();
          if ($aiAjax.length) {
            const targetId = $aiAjax.attr('data-ai-ajax');
            if (targetId) {
              const root = document.getElementById(targetId);
              if (root) {
                $resultsBlock = $(root).find('.ai-search-block-output');
                $suffixText = $(root).find('.suffix_text');
              }
            }
          }

          // Fallback to the global response block if no per-block
          // target was found. Do NOT create elements — rely on the
          // page author to place the response block alongside the form.
          if (!$resultsBlock || $resultsBlock.length === 0) {
            $resultsBlock = $(
              '#ai-search-block-response .ai-search-block-output',
            );
            $suffixText = $('#ai-search-block-response .suffix_text');
          }

          // Ensure a DB results container exists after the AI output.
          let $dbResults = $('#ai-search-block-db-results');
          if (!$dbResults.length) {
            $dbResults = $('<div id="ai-search-block-db-results"></div>');
            $resultsBlock.after($dbResults);
          } else {
            $dbResults.empty();
          }

          // One-time CSS for hiding exposed form & focus outline in DB results.
          if (!document.getElementById('ai-hide-exposed')) {
            const style = document.createElement('style');
            style.id = 'ai-hide-exposed';
            style.textContent = `
              #ai-search-block-db-results .views-exposed-form,
              #ai-db-results-block .views-exposed-form,
              .ai-db-results .views-exposed-form,
              #ai-search-block-db-results .view-empty,
              #ai-db-results-block .view-empty,
              .ai-db-results .view-empty { display: none !important; }
              #ai-search-block-db-results:focus,
              #ai-db-results-block:focus,
              .ai-db-results:focus { outline: 0 !important; box-shadow: none !important; }
              `;
            document.head.appendChild(style);
          }

          // Handle Enter key submission for textarea if enabled.
          const inputField = drupalSettings.ai_search_block?.input_field;
          const submitOnEnter = drupalSettings.ai_search_block?.submit_on_enter;

          if (inputField === 'textarea' && submitOnEnter) {
            const $queryField = $form.find(
              '[data-drupal-selector="edit-query"]',
            );
            $queryField.on('keydown', function (e) {
              // Check if Enter was pressed without Shift
              if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                e.stopPropagation();
                e.stopImmediatePropagation();
                $form.trigger('submit');
                return false;
              }
            });
          }

          // Hide suffix text initially.
          if ($suffixText.length) {
            $suffixText.hide();
          }

          // Determine loading message and create a reusable loader element.
          const loadingMsg =
            drupalSettings.ai_search_block &&
            drupalSettings.ai_search_block.loading_text
              ? drupalSettings.ai_search_block.loading_text
              : 'Loading...';

          // Check if loading message contains multiple messages separated by |
          let loadingMessages = [];
          let hasMultipleMessages = false;
          let loadingMessageInterval = null;

          if (loadingMsg.includes('|')) {
            loadingMessages = loadingMsg.split('|').map(msg => msg.trim()).filter(msg => msg.length > 0);
            hasMultipleMessages = loadingMessages.length > 1;
          }

          const $loader = $(
            `<p class="loading_text"><span class="loader"></span><span class="loading-message">${
              hasMultipleMessages ? loadingMessages[0] : loadingMsg
            }</span></p>`,
          );

          // Function to start alternating loading messages
          function startLoadingMessageRotation() {
            if (!hasMultipleMessages) return;

            let currentIndex = 0;
            loadingMessageInterval = setInterval(function() {
              currentIndex = (currentIndex + 1) % loadingMessages.length;
              $loader.find('.loading-message').text(loadingMessages[currentIndex]);
            }, 2000);
          }

          // Function to stop alternating loading messages
          function stopLoadingMessageRotation() {
            if (loadingMessageInterval) {
              clearInterval(loadingMessageInterval);
              loadingMessageInterval = null;
            }
          }
          
          // --- Helpers for DB results (minimal additions) ---
          function getScrollOffset() {
            let extra = 0;
            const $toolbar = $('#toolbar-bar:visible');
            if ($toolbar.length) {
              extra += $toolbar.outerHeight();
            }
            const $fixedHeader = $(
              '.site-header.is-fixed:visible, header.fixed:visible',
            );
            if ($fixedHeader.length) {
              extra += $fixedHeader.outerHeight();
            }
            const cfg =
              drupalSettings.ai_search_block &&
              drupalSettings.ai_search_block.scroll_offset;
            if (typeof cfg === 'number') {
              extra += cfg;
            }
            return extra;
          }

          function scrollToResults($target) {
            if (!$target || !$target.length) {
              return;
            }
            const top = Math.max(
              0,
              $target.offset().top - getScrollOffset() - 12,
            );
            $('html, body').scrollTop(top);
            $target.attr('tabindex', '-1').trigger('focus');
          }

          function hideExposedForm($target) {
            const $exposed = $target.find('form.views-exposed-form');
            if (!$exposed.length) {
              return;
            }
            $exposed.attr('aria-hidden', 'true').addClass('ai-exposed-hidden');
            $exposed[0].style.display = 'none';
            $exposed
              .find('input,select,textarea,button,a')
              .attr('tabindex', '-1');
          }

          function fixPager($target, currentPage) {
            const $ul = $target
              .find('ul.pagination, ul.js-pager__items')
              .first();
            if (!$ul.length) {
              return;
            }

            $target.data('currentPage', currentPage);

            $ul
              .find('li.page-item')
              .removeClass('active')
              .each(function itemEach() {
                const $li = $(this);
                const $span = $li.find('> span.page-link');
                if ($span.length) {
                  const label = $span.html();
                  $span.replaceWith(
                    $('<a class="page-link" href="#"></a>').html(label),
                  );
                }
              });

            const targetLabel = (currentPage + 1).toString();
            const $liForPage = $ul
              .find('li.page-item')
              .filter(function itemFilter() {
                const $el = $(this)
                  .find('> a.page-link, > span.page-link')
                  .first();
                const text = ($el.html() || '').trim();
                return /^\d+$/.test(text) && text === targetLabel;
              })
              .first();

            if ($liForPage.length) {
              $liForPage.addClass('active');
              const $a = $liForPage.find('> a.page-link');
              if ($a.length) {
                $a.replaceWith(
                  $('<span class="page-link"></span>').html(targetLabel),
                );
              }
            }

            $ul.find('li.page-item').each(function pageItemEach() {
              const $li = $(this);
              const $el = $li.find('> a.page-link, > span.page-link').first();
              const txt = ($el.html() || '').trim();

              if (/^\d+$/.test(txt)) {
                const n = Number.parseInt(txt, 10) - 1;
                $li.attr('data-page', n);
                if ($el[0] && $el[0].tagName === 'A') {
                  $el.attr('href', '#').attr('role', 'button');
                }
              } else {
                $li.removeAttr('data-page');
              }
            });

            $ul.find('li.page-item[data-synth]').remove();
            if (currentPage > 0) {
              const $prevLi = $(
                `<li class="page-item" data-synth="prev">
<a class="page-link" href="#" role="button" title="Go to previous page">
<span aria-hidden="true">‹‹</span><span class="visually-hidden">Previous page</span></a></li>`,
              );

              const $firstLi = $(
                `<li class="page-item" data-synth="first">
<a class="page-link" href="#" role="button" title="Go to first page">
<span aria-hidden="true">« First</span><span class="visually-hidden">First page</span></a></li>`,
              );

              const $firstNumeric = $ul
                .find('li.page-item')
                .filter(function numericFilter() {
                  const t = $(this)
                    .find('> a.page-link, > span.page-link')
                    .first()
                    .html()
                    .trim();
                  return /^\d+$/.test(t);
                })
                .first();

              if ($firstNumeric.length) {
                $firstNumeric.before($prevLi).before($firstLi);
              } else {
                $ul.prepend($firstLi).prepend($prevLi);
              }
            }
          }

          function fetchDbResults(page, queryVal, blockIdVal) {
            const pageNum = typeof page === 'undefined' ? 0 : page;

            const seq =
              (Number.parseInt($dbResults.data('reqSeq'), 10) || 0) + 1;
            $dbResults.data('reqSeq', seq);

            aiSearchBlockConsoleLog(
              'AI Search: REQUEST → DB results, requested page:',
              pageNum,
              'seq:',
              seq,
            );
            dispatchStatusEvent('page-requested', { page: pageNum, seq });

            if (
              !drupalSettings.ai_search_block ||
              !drupalSettings.ai_search_block.enable_database_results
            ) {
              aiSearchBlockConsoleLog('AI Search: DB results disabled in settings');
              return;
            }

            $dbResults.html(
              '<p class="loading_text">Loading database results...</p>',
            );

            $.ajax({
              url:
                (drupalSettings.ai_search_block &&
                  drupalSettings.ai_search_block.db_results_url) ||
                '/ai-search-block/db-results',
              type: 'POST',
              data: {
                query: queryVal,
                block_id: blockIdVal,
                page: pageNum,
              },
              success(data) {
                const latestSeq =
                  Number.parseInt($dbResults.data('reqSeq'), 10) || 0;
                if (seq !== latestSeq) {
                  aiSearchBlockConsoleLog(
                    `AI Search: STALE response ignored (seq${seq} latest${latestSeq})`,
                  );
                  return;
                }

                aiSearchBlockConsoleLog(
                  'AI Search: RESPONSE ← DB results, requested page:',
                  pageNum,
                  'seq:',
                  seq,
                );

                if (data && data.html) {
                  $dbResults.html(data.html);

                  // Remove use-ajax to avoid core AJAX handling.
                  $dbResults.find('a.use-ajax').removeClass('use-ajax');

                  // Ensure Views responsive grid CSS is present.
                  if (
                    !$('link[href*="views-responsive-grid.css"]').length &&
                    drupalSettings.path &&
                    drupalSettings.path.baseUrl !== undefined
                  ) {
                    const link = document.createElement('link');
                    link.rel = 'stylesheet';
                    link.href = `${drupalSettings.path.baseUrl}core/modules/views/css/views-responsive-grid.css`;
                    document.head.appendChild(link);
                  }

                  Drupal.attachBehaviors($dbResults[0]);

                  fixPager($dbResults, pageNum);

                  $dbResults.data('currentPage', pageNum);
                  aiSearchBlockConsoleLog(
                    'AI Search: LOADED/APPLIED page:',
                    $dbResults.data('currentPage'),
                  );

                  hideExposedForm($dbResults);

                  scrollToResults($dbResults);

                  $dbResults
                    .off('click.aiPager')
                    .on('click.aiPager', 'a.page-link', function pagerClick(e) {
                      e.preventDefault();
                      e.stopPropagation();
                      e.stopImmediatePropagation();

                      const $a = $(this);
                      const $li = $a.closest('li.page-item');
                      const cur =
                        Number.parseInt($dbResults.data('currentPage'), 10) ||
                        0;
                      let nextPage;

                      if ($li.attr('data-page')) {
                        nextPage = Number.parseInt($li.attr('data-page'), 10);
                      } else if ($li.attr('data-synth') === 'first') {
                        nextPage = 0;
                      } else if ($li.attr('data-synth') === 'prev') {
                        nextPage = Math.max(0, cur - 1);
                      } else {
                        const label = ($a.html() || '').trim().toLowerCase();
                        if (
                          label.includes('next') ||
                          label === '›' ||
                          label === '››'
                        ) {
                          nextPage = cur + 1;
                        } else if (
                          label.includes('prev') ||
                          label === '‹' ||
                          label === '‹‹'
                        ) {
                          nextPage = Math.max(0, cur - 1);
                        }
                      }

                      if (
                        typeof nextPage === 'number' &&
                        !Number.isNaN(nextPage)
                      ) {
                        aiSearchBlockConsoleLog(
                          'AI Search: REQUESTED page (via click):',
                          nextPage,
                          'current:',
                          cur,
                        );
                        fetchDbResults(nextPage, queryVal, blockIdVal);
                        scrollToResults($dbResults);
                        return false;
                      }
                      return true;
                    });
                } else {
                  $dbResults.html('<p>No database results found.</p>');
                }
              },
              error(jqXHR, textStatus, errorThrown) {
                const latestSeq =
                  Number.parseInt($dbResults.data('reqSeq'), 10) || 0;
                aiSearchBlockConsoleLog(
                  'AI Search: DB results ERROR for requested page:',
                  pageNum,
                  'seq:',
                  seq,
                  textStatus,
                  errorThrown,
                  'latest seq:',
                  latestSeq,
                );
                $dbResults.html('<p>Error loading database results.</p>');
              },
            });
          }
          // ---------------- End Helpers ------------------

          once('aiSearchSubmit', $form).forEach(() => {
            $form.on('submit', function (event) {
              aiSearchBlockConsoleLog('Submit handler triggered', new Date().getTime());
              event.preventDefault();

              const submitButton = $form.find(
                '[data-drupal-selector="edit-submit"]',
              );
              submitButton.prop('disabled', true);

              // Show the loader initially.
              $resultsBlock.html($loader);
              $dbResults.empty();

              // Start loading message rotation if applicable
              startLoadingMessageRotation();

              // Dispatch loading status
              dispatchStatusEvent('loading', { form: $form[0] });

              // Retrieve form values.
              const queryVal =
                document.querySelector('[data-drupal-selector="edit-query"]')
                  .value || '';
              const streamVal =
                document.querySelector('[data-drupal-selector="edit-stream"]')
                  .value || '';
              const blockIdVal =
                document.querySelector('[data-drupal-selector="edit-block-id"]')
                  .value || '';

              // Clear the form if clear_after_search is enabled.
              const clearAfterSearch =
                drupalSettings.ai_search_block.clear_after_search;
              if (clearAfterSearch) {
                const queryField = document.querySelector(
                  '[data-drupal-selector="edit-query"]',
                );
                if (queryField) {
                  queryField.value = '';
                }
              }

              // If streaming is enabled (using '1' as true).
              if (streamVal === 'true') {
                $suffixText.hide();
                try {
                  const xhr = new XMLHttpRequest();
                  xhr.open(
                    'POST',
                    drupalSettings.ai_search_block.submit_url,
                    true,
                  );
                  xhr.setRequestHeader('Content-Type', 'application/json');
                  xhr.setRequestHeader('Accept', 'application/json');

                  // Cache variables to hold the full output.
                  let lastResponseLength = 0;
                  let joined = '';

                  xhr.onprogress = function () {
                    const responseText = xhr.responseText || '';
                    // Get only the new part of the response.
                    const newData = responseText.substring(lastResponseLength);
                    lastResponseLength = responseText.length;

                    // Split new data using the delimiter.
                    const chunks = newData.trim().split('|§|').filter(Boolean);

                    // Parse each chunk and accumulate the answer pieces.
                    chunks.forEach(function (chunk) {
                      try {
                        const parsed = JSON.parse(chunk);
                        // Update log Id from each chunk.
                        if (parsed.log_id) {
                          drupalSettings.ai_search_block.logId = parsed.log_id;
                        }
                        joined += parsed.answer_piece || '';
                      } catch (e) {
                        aiSearchBlockConsoleLog('Error parsing chunk:', e, chunk);
                      }
                    });

                    // Overwrite the full output (letting browsers fix broken HTML)
                    // and re-append the loader.
                    $resultsBlock.html(joined).append($loader);

                    // Dispatch streaming progress status
                    dispatchStatusEvent('streaming', {
                      progress: joined.length,
                      form: $form[0],
                    });
                  };

                  xhr.onreadystatechange = function () {
                    if (xhr.readyState === 4) {
                      if (xhr.status === 200) {
                        // Stop loading message rotation and remove the loader
                        stopLoadingMessageRotation();
                        $loader.remove();
                        submitButton.prop('disabled', false);
                        if ($suffixText.length) {
                          $suffixText.html(
                            drupalSettings.ai_search_block.suffix_text,
                          );
                          $suffixText.show();
                          Drupal.attachBehaviors($suffixText[0]);
                        }
                        // Dispatch done status
                        dispatchStatusEvent('done', {
                          response: joined,
                          form: $form[0],
                        });

                        // Fetch DB results after successful streaming response.
                        if (
                          drupalSettings.ai_search_block &&
                          drupalSettings.ai_search_block.enable_database_results
                        ) {
                          aiSearchBlockConsoleLog(
                            'AI Search: Fetching DB results after stream',
                          );
                          $dbResults.data('currentPage', 0);
                          fetchDbResults(0, queryVal, blockIdVal);
                        }
                      } else if (xhr.status === 500) {
                        // Stop loading message rotation on error
                        stopLoadingMessageRotation();
                        $resultsBlock.html('An error happened.');
                        aiSearchBlockConsoleLog('Error response:', xhr.responseText);
                        submitButton.prop('disabled', false);
                        // Dispatch error status
                        dispatchStatusEvent('error', {
                          error: xhr.responseText,
                          form: $form[0],
                        });
                        try {
                          const parsedError = JSON.parse(xhr.responseText);
                          if (
                            parsedError.response &&
                            parsedError.response.answer_piece
                          ) {
                            $resultsBlock.html(parsedError.response.answer_piece);
                          }
                          Drupal.attachBehaviors($resultsBlock[0]);
                        } catch (e) {
                          aiSearchBlockConsoleLog('Error parsing 500 response:', e);
                        }
                      }
                    }
                  };

                  // Send the streaming request.
                  xhr.send(
                    JSON.stringify({
                      query: queryVal,
                      stream: streamVal,
                      block_id: blockIdVal,
                    }),
                  );
                } catch (e) {
                  // Stop loading message rotation on exception
                  stopLoadingMessageRotation();
                  aiSearchBlockConsoleLog('XHR error:', e);
                  dispatchStatusEvent('error', { error: e, form: $form[0] });
                }
              } else {
                $suffixText.hide();
                // Non-streaming: use jQuery.post.
                $.post(
                  drupalSettings.ai_search_block.submit_url,
                  {
                    query: queryVal,
                    stream: streamVal,
                    block_id: blockIdVal,
                  },
                  function (data) {
                    // Stop loading message rotation on success
                    stopLoadingMessageRotation();
                    if (data && data.response) {
                      $resultsBlock.html(data.response);
                    }
                    // Set log Id if available.
                    if (data && data.log_id) {
                      drupalSettings.ai_search_block.logId = data.log_id;
                    }
                    if ($suffixText.length) {
                      $suffixText
                        .html(drupalSettings.ai_search_block.suffix_text)
                        .show();
                      Drupal.attachBehaviors($suffixText[0]);
                    }
                    submitButton.prop('disabled', false);
                    dispatchStatusEvent('done', {
                      response: data,
                      form: $form[0],
                    });
                  },
                )
                  .fail(function (jqXHR) {
                    // Stop loading message rotation on error
                    stopLoadingMessageRotation();
                    $resultsBlock.html('An error happened.');
                    aiSearchBlockConsoleLog('Error on non-streaming request');
                    dispatchStatusEvent('error', {
                      error: jqXHR.responseText,
                      form: $form[0],
                    });
                  })
                  .always(function () {
                    // Fetch DB results after non-streaming request completes.
                    if (
                      drupalSettings.ai_search_block &&
                      drupalSettings.ai_search_block.enable_database_results
                    ) {
                      aiSearchBlockConsoleLog('AI Search: Fetching DB results after POST');
                      $dbResults.data('currentPage', 0);
                      fetchDbResults(0, queryVal, blockIdVal);
                    }
                    submitButton.prop('disabled', false);
                  });
              }

              return false;
            });
          });
        },
      );
    },
    clipboard(context, settings){
      // Handle copy to clipboard functionality
      $('.ai-search-block-copy-output').on('click', function (e) {
        e.preventDefault();
        // Find the related output div
        const $outputDiv = $(this).closest('#ai-search-block-response').find('.ai-search-block-output');
        if ($outputDiv.length) {
          const plain = $outputDiv[0].textContent.trim();
          const html = $outputDiv.html().trim();
          if (navigator.clipboard && navigator.clipboard.write) {
            navigator.clipboard.write([
              new ClipboardItem({
                'text/html': new Blob([html], { type: 'text/html' }),
                'text/plain': new Blob([plain], { type: 'text/plain' })
              })
            ])
              .then(function () {
                const $link = e.target;
                const old = $link.textContent;
                $link.textContent = 'Copied!';
                setTimeout(function () {
                  $link.textContent = old;
                }, 2000);
              })
              .catch(function (err) {
                aiSearchBlockConsoleLog('Copy failed', err);
                alert('Failed to copy');
              });
          } else {
            aiSearchBlockConsoleLog('Clipboard API not supported');
            alert('Your browser does not support rich copy');
          }
        } else {
          aiSearchBlockConsoleLog('Output div not found');
        }
      })
    },

    feedback(context, settings){
      // Create feedback modal if it doesn't exist
      if (!$('#ai-search-block-feedback-modal').length) {
        const modalHtml = `
          <div id="ai-search-block-feedback-modal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:9999; align-items:center; justify-content:center;">
            <div style="background:white; padding:20px; border-radius:8px; max-width:500px; width:90%; box-shadow:0 2px 10px rgba(0,0,0,0.3);">
              <h3 style="margin-top:0;">${Drupal.t('Help us improve, what was wrong?')}</h3>
              <form id="ai-search-block-feedback-form">
                <textarea id="ai-search-block-feedback-text" style="width:100%; min-height:100px; padding:8px; border:1px solid #ccc; border-radius:4px; font-family:inherit; font-size:14px; margin-bottom:10px;" placeholder="${Drupal.t('Please tell us what was wrong...')}"></textarea>
                <div style="display:flex; gap:10px; justify-content:flex-end;">
                  <button type="button" id="ai-search-block-feedback-cancel" style="padding:8px 16px; border:1px solid #ccc; background:white; border-radius:4px; cursor:pointer;">${Drupal.t('Cancel')}</button>
                  <button type="submit" id="ai-search-block-feedback-submit" style="padding:8px 16px; border:none; background:#0073aa; color:white; border-radius:4px; cursor:pointer;">${Drupal.t('Submit')}</button>
                </div>
              </form>
            </div>
          </div>
        `;
        $('body').append(modalHtml);
      }

      const $modal = $('#ai-search-block-feedback-modal');
      const $form = $('#ai-search-block-feedback-form');
      const $textarea = $('#ai-search-block-feedback-text');

      // Handle feedback link clicks
      $(document).on('click', '.ai-search-block-open-feedback', function (e) {
        e.preventDefault();
        const logId = drupalSettings.ai_search_block?.logId;

        if (!logId) {
          alert(Drupal.t('Unable to submit feedback at this time.'));
          return;
        }

        // Store logId on the modal for later use
        $modal.data('logId', logId);

        // Clear previous feedback text
        $textarea.val('');

        // Show modal
        $modal.css('display', 'flex');
        $textarea.focus();
      });

      // Handle cancel button
      $('#ai-search-block-feedback-cancel').on('click', function () {
        $modal.css('display', 'none');
      });

      // Handle modal background click to close
      $modal.on('click', function (e) {
        if (e.target.id === 'ai-search-block-feedback-modal') {
          $modal.css('display', 'none');
        }
      });

      // Handle form submission
      $form.on('submit', function (e) {
        e.preventDefault();

        const feedbackText = $textarea.val().trim();
        const logId = $modal.data('logId');

        if (!feedbackText) {
          alert(Drupal.t('Please enter some feedback.'));
          return;
        }

        // Submit feedback via AJAX
        $.post('/ai-search-block-log/score', {
          log_id: logId,
          feedback: feedbackText
        })
        .done(function (response) {
          // Hide modal on success
          $modal.css('display', 'none');

          // Optionally show a success message
          if (response.response) {
            aiSearchBlockConsoleLog('Feedback submitted:', response.response);
          }
        })
        .fail(function (jqXHR) {
          aiSearchBlockConsoleLog('Error submitting feedback:', jqXHR.responseText);
          alert(Drupal.t('Failed to submit feedback. Please try again.'));
        });
      });
    },

    animatePlaceholder(context, settings) {
      // Check if animation is enabled
      if (!drupalSettings.ai_search_block || !drupalSettings.ai_search_block.animate_placeholder) {
        return;
      }

      const placeholder = drupalSettings.ai_search_block.placeholder || '';

      // Check if placeholder contains multiple texts separated by |
      if (!placeholder.includes('|')) {
        return;
      }

      // Split placeholders and trim whitespace
      const placeholders = placeholder.split('|').map(text => text.trim()).filter(text => text.length > 0);

      if (placeholders.length <= 1) {
        return;
      }

      // Find the input field
      const $input = $(context).find('[data-drupal-selector="edit-query"]');

      if (!$input.length) {
        return;
      }

      let currentIndex = 0;
      let charIndex = 0;
      let isTyping = true;
      let isUserInteracting = false;
      let animationTimeout = null;

      // Stop animation if user interacts with the field
      $input.on('focus click keydown', function() {
        isUserInteracting = true;
        if (animationTimeout) {
          clearTimeout(animationTimeout);
        }
      });

      // Resume animation if user leaves the field empty
      $input.on('blur', function() {
        if ($input.val() === '') {
          isUserInteracting = false;
          charIndex = 0;
          isTyping = true;
          animate();
        }
      });

      function animate() {
        // Stop if user is interacting
        if (isUserInteracting) {
          return;
        }

        const currentText = placeholders[currentIndex];

        if (isTyping) {
          // Typing phase
          if (charIndex < currentText.length) {
            $input.attr('placeholder', currentText.substring(0, charIndex + 1));
            charIndex++;
            animationTimeout = setTimeout(animate, 100); // Type speed: 100ms per character
          } else {
            // Finished typing, pause before erasing
            isTyping = false;
            animationTimeout = setTimeout(animate, 2000); // Pause for 2 seconds
          }
        } else {
          // Erasing phase
          if (charIndex > 0) {
            charIndex--;
            $input.attr('placeholder', currentText.substring(0, charIndex));
            animationTimeout = setTimeout(animate, 50); // Erase speed: 50ms per character
          } else {
            // Finished erasing, move to next placeholder
            currentIndex = (currentIndex + 1) % placeholders.length;
            isTyping = true;
            animationTimeout = setTimeout(animate, 500); // Pause before typing next
          }
        }
      }

      // Start the animation
      animate();
    }
  };
})(jQuery, Drupal, drupalSettings, once);
