<?php

namespace Drupal\ai_search_block_log;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The helper class.
 */
class AiSearchBlockLogHelper implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
    );
  }

  /**
   * Delete the expired logs.
   *
   * @return void
   *   Returns nothing.
   */
  public function cron(): void {
    $now = time();
    $ids = $this->entityTypeManager->getStorage('ai_search_block_log')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('expiry', $now, '<')
      ->execute();

    $this->entityTypeManager->getStorage('ai_search_block_log')
      ->delete(array_values($ids));
  }

  /**
   * Open the log line.
   *
   * @param string $block_id
   *   The block.
   * @param int $uid
   *   THe user id.
   * @param string $query
   *   The question.
   *
   * @return int|mixed|string|null
   *   Returns the log item id.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function start(string $block_id, int $uid, string $query): mixed {
    $storage = $this->entityTypeManager->getStorage('ai_search_block_log');
    $expiry = $this->configFactory->get('ai_search_block_log.settings')
      ->get('expiry');
    $expiry = $expiry ?? 'week';

    /** @var \Drupal\ai_search_block_log\Entity\AiSearchBlockLog $log */
    $log = $storage->create([
      'uid' => $uid,
      'block_id' => $block_id,
      'created' => time(),
      'expiry' => strtotime('now + 1 ' . $expiry),
      'question' => [
        'value' => $query,
        'format' => 'plain_text',
      ],
    ]);
    $log->save();

    return $log->id();
  }

  /**
   * Log the response to the DB.
   *
   * @param int $id
   *   The id to log it in.
   * @param string $response
   *   The response to log.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function logResponse(int $id, string $response): void {
    /** @var \Drupal\ai_search_block_log\Entity\AiSearchBlockLog|null $entity */
    $entity = $this->entityTypeManager
      ->getStorage('ai_search_block_log')
      ->load($id);
    if (!$entity) {
      return;
    }
    $entity->set('response_given', $response);
    $entity->save();
  }

  /**
   * Update the log with fields.
   *
   * @param int $id
   *   The id of the log item to update.
   * @param array<string, mixed> $fields
   *   The fields to update.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function update(int $id, array $fields): void {
    /** @var \Drupal\ai_search_block_log\Entity\AiSearchBlockLog|null $entity */
    $entity = $this->entityTypeManager
      ->getStorage('ai_search_block_log')
      ->load($id);
    if (!$entity) {
      return;
    }
    foreach ($fields as $key => $field) {
      $entity->set($key, $field);
    }
    $entity->save();
  }

}
