(function ($, Drupal, drupalSettings, once) {
  // Utility to dispatch custom status events
  function dispatchStatusEvent(status, extra) {
    const event = new CustomEvent('ai-search-status', {
      detail: { status, ...(extra || {}) },
    });
    document.dispatchEvent(event);
  }

  Drupal.behaviors.aiSearchBlock = {
    attach(context, settings) {
      // Attach the submit handler only once per form.
      once('aiSearchForm', '.ai-search-block-form', context).forEach(
        function (formElem) {
          console.log('Attaching behavior to:', formElem);
          const $form = $(formElem);

          // Locate the results output region and the suffix text.
          const $resultsBlock = $(
            '#ai-search-block-response .ai-search-block-output',
          );
          const $suffixText = $('#ai-search-block-response .suffix_text');

          if (!$resultsBlock.length) {
            console.warn(
              'AI Search: Could not find a results block relative to the form.',
            );
            return;
          }

          // Handle Enter key submission for textarea if enabled.
          const inputField = drupalSettings.ai_search_block?.input_field;
          const submitOnEnter = drupalSettings.ai_search_block?.submit_on_enter;

          if (inputField === 'textarea' && submitOnEnter) {
            const $queryField = $form.find('[data-drupal-selector="edit-query"]');

            $queryField.on('keydown', function (e) {
              // Check if Enter was pressed without Shift
              if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                $form.trigger('submit');
              }
            });
          }

          // Hide suffix text initially.
          if ($suffixText.length) {
            $suffixText.hide();
          }

          // Determine loading message and create a reusable loader element.
          const loadingMsg =
            drupalSettings.ai_search_block &&
            drupalSettings.ai_search_block.loading_text
              ? drupalSettings.ai_search_block.loading_text
              : 'Loading...';
          const $loader = $(
            `<p class="loading_text"><span class="loader"></span>${
              loadingMsg
            }</p>`,
          );

          $form.on('submit', function (event) {
            console.log('Submit handler triggered', new Date().getTime());
            event.preventDefault();

            const submitButton = $form.find(
              '[data-drupal-selector="edit-submit"]',
            );
            submitButton.prop('disabled', true);

            // Show the loader initially.
            $resultsBlock.html($loader);

            // Dispatch loading status
            dispatchStatusEvent('loading', { form: $form[0] });

            // Retrieve form values.
            const queryVal =
              document.querySelector('[data-drupal-selector="edit-query"]')
                .value || '';
            const streamVal =
              document.querySelector('[data-drupal-selector="edit-stream"]')
                .value || '';
            const blockIdVal =
              document.querySelector('[data-drupal-selector="edit-block-id"]')
                .value || '';

            // If streaming is enabled (using '1' as true).
            if (streamVal === 'true') {
              try {
                const xhr = new XMLHttpRequest();
                xhr.open(
                  'POST',
                  drupalSettings.ai_search_block.submit_url,
                  true,
                );
                xhr.setRequestHeader('Content-Type', 'application/json');
                xhr.setRequestHeader('Accept', 'application/json');

                // Cache variables to hold the full output.
                let lastResponseLength = 0;
                let joined = '';

                xhr.onprogress = function () {
                  const responseText = xhr.responseText || '';
                  // Get only the new part of the response.
                  const newData = responseText.substring(lastResponseLength);
                  lastResponseLength = responseText.length;

                  // Split new data using the delimiter.
                  const chunks = newData.trim().split('|§|').filter(Boolean);

                  // Parse each chunk and accumulate the answer pieces.
                  chunks.forEach(function (chunk) {
                    try {
                      const parsed = JSON.parse(chunk);
                      // Update log Id from each chunk.
                      if (parsed.log_id) {
                        drupalSettings.ai_search_block.logId = parsed.log_id;
                      }
                      joined += parsed.answer_piece || '';
                    } catch (e) {
                      console.error('Error parsing chunk:', e, chunk);
                    }
                  });

                  // Overwrite the full output (letting browsers fix broken HTML)
                  // and re-append the loader.
                  $resultsBlock.html(joined).append($loader);

                  // Dispatch streaming progress status
                  dispatchStatusEvent('streaming', {
                    progress: joined.length,
                    form: $form[0],
                  });
                };

                xhr.onreadystatechange = function () {
                  if (xhr.readyState === 4) {
                    if (xhr.status === 200) {
                      // Remove the loader upon successful completion.
                      $loader.remove();
                      submitButton.prop('disabled', false);
                      if ($suffixText.length) {
                        $suffixText.html(
                          drupalSettings.ai_search_block.suffix_text,
                        );
                        Drupal.attachBehaviors($suffixText[0]);
                        $suffixText.show();
                      }
                      // Dispatch done status
                      dispatchStatusEvent('done', {
                        response: joined,
                        form: $form[0],
                      });
                    } else if (xhr.status === 500) {
                      $resultsBlock.html('An error happened.');
                      console.error('Error response:', xhr.responseText);
                      submitButton.prop('disabled', false);
                      // Dispatch error status
                      dispatchStatusEvent('error', {
                        error: xhr.responseText,
                        form: $form[0],
                      });
                      try {
                        const parsedError = JSON.parse(xhr.responseText);
                        if (
                          parsedError.response &&
                          parsedError.response.answer_piece
                        ) {
                          $resultsBlock.html(parsedError.response.answer_piece);
                        }
                        Drupal.attachBehaviors($resultsBlock[0]);
                      } catch (e) {
                        console.error('Error parsing 500 response:', e);
                      }
                    }
                  }
                };

                // Send the streaming request.
                xhr.send(
                  JSON.stringify({
                    query: queryVal,
                    stream: streamVal,
                    block_id: blockIdVal,
                  }),
                );
              } catch (e) {
                console.error('XHR error:', e);
                dispatchStatusEvent('error', { error: e, form: $form[0] });
              }
            } else {
              // Non-streaming: use jQuery.post.
              $.post(
                drupalSettings.ai_search_block.submit_url,
                {
                  query: queryVal,
                  stream: streamVal,
                  block_id: blockIdVal,
                },
                function (data) {
                  if (data && data.response) {
                    $resultsBlock.html(data.response);
                  }
                  // Set log Id if available.
                  if (data && data.log_id) {
                    drupalSettings.ai_search_block.logId = data.log_id;
                  }
                  if ($suffixText.length) {
                    $suffixText
                      .html(drupalSettings.ai_search_block.suffix_text)
                      .show();
                    Drupal.attachBehaviors($suffixText[0]);
                  }
                  submitButton.prop('disabled', false);
                  dispatchStatusEvent('done', {
                    response: data,
                    form: $form[0],
                  });
                },
              )
                .fail(function (jqXHR) {
                  $resultsBlock.html('An error happened.');
                  console.error('Error on non-streaming request');
                  dispatchStatusEvent('error', {
                    error: jqXHR.responseText,
                    form: $form[0],
                  });
                })
                .always(function () {
                  submitButton.prop('disabled', false);
                });
            }

            return false;
          });
        },
      );
    },
  };
})(jQuery, Drupal, drupalSettings, once);
