<?php

namespace Drupal\ai_search_block\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Drupal\ai_search_block\AiSearchBlockHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * An example controller.
 */
class AiSearchBlockController extends ControllerBase {

  /**
   * The block storage.
   *
   * @var \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   */
  protected $blockStorage;

  /**
   * Constructor.
   *
   * @param \Drupal\ai_search_block\AiSearchBlockHelper $searchBlockHelper
   *   The form builder.
   */
  public function __construct(
    protected AiSearchBlockHelper $searchBlockHelper,
  ) {
    $this->blockStorage = $this->entityTypeManager()->getStorage('block');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): AiSearchBlockController {
    return new static(
      $container->get('ai_search_block.helper'),
    );
  }

  /**
   * Returns a renderable array for a test page.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|string|\Symfony\Component\HttpFoundation\StreamedResponse
   *   The response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function search(Request $request): JsonResponse|string|StreamedResponse {
    if ($request->query->get('block_id')) {
      $query = $request->query->get('query');
      $block_id = $request->query->get('block_id');
      $stream = $request->query->get('stream');
    }
    else {
      $input = file_get_contents('php://input');
      $data = Json::decode($input ?: '{}');
      $query = $data['query'] ?? NULL;
      $stream = $data['stream'] ?? NULL;
      $block_id = $data['block_id'] ?? NULL;
    }
    if (empty($block_id)) {
      return new JsonResponse(
        [
          'response' => 'Missing required parameter: block_id',
          'status' => 'error',
        ],
        400
      );
    }
    /** @var \Drupal\block\BlockInterface|null $block */
    $block = $this->blockStorage->load($block_id);
    $logId = 0;
    if (function_exists('ai_search_block_log_start')) {
      $logId = ai_search_block_log_start($block_id, $this->currentUser()->id(),
        $query);
    }
    if ($block) {
      $settings = $block->get('settings');
      $this->searchBlockHelper->setConfig($settings);
      $this->searchBlockHelper->setBlockId($block_id);
      $this->searchBlockHelper->logId = $logId;
      $results = $this->searchBlockHelper->searchRagAction($query);

      if ($stream === "true" || $stream === "TRUE") {
        header('X-Accel-Buffering: no');
        // Making maximum execution time unlimited.
        set_time_limit(0);
        ob_implicit_flush();

        return $results;
      }

      return $results;
    }

    if (function_exists('ai_search_block_log_add_response')) {
      ai_search_block_log_add_response($logId, 'There was an error fetching your data');
    }

    return new JsonResponse([
      'response' => 'There was an error fetching your data',
      'log_id' => $logId,
    ]);
  }

}
