(function ($, Drupal, drupalSettings, once) {
  // Utility to dispatch custom status events
  function dispatchStatusEvent(status, extra) {
    const event = new CustomEvent('ai-search-status', {
      detail: { status, ...(extra || {}) },
    });
    document.dispatchEvent(event);
  }

  Drupal.behaviors.aiSearchBlock = {
    attach(context, settings) {
      // Attach the submit handler only once per form.
      once('aiSearchForm', '.ai-search-block-form', context).forEach(
        function (formElem) {
          console.log('Attaching behavior to:', formElem);
          const $form = $(formElem);

          // Locate the results output region and the suffix text.
          // Prefer the per-block response target set on the submit
          // button (`data-ai-ajax`). This ensures we use the response
          // block that matches this form instance when multiple blocks
          // are present on the page.
          let $resultsBlock = $();
          let $suffixText = $();
          const $aiAjax = $form.find('[data-ai-ajax]').first();
          if ($aiAjax.length) {
            const targetId = $aiAjax.attr('data-ai-ajax');
            if (targetId) {
              const root = document.getElementById(targetId);
              if (root) {
                $resultsBlock = $(root).find('.ai-search-block-output');
                $suffixText = $(root).find('.suffix_text');
              }
            }
          }

          // Fallback to the global response block if no per-block
          // target was found. Do NOT create elements — rely on the
          // page author to place the response block alongside the form.
          if (!$resultsBlock || $resultsBlock.length === 0) {
            $resultsBlock = $(
              '#ai-search-block-response .ai-search-block-output',
            );
            $suffixText = $('#ai-search-block-response .suffix_text');
          }

          // Ensure a DB results container exists after the AI output.
          let $dbResults = $('#ai-search-block-db-results');
          if (!$dbResults.length) {
            $dbResults = $('<div id="ai-search-block-db-results"></div>');
            $resultsBlock.after($dbResults);
          } else {
            $dbResults.empty();
          }

          // One-time CSS for hiding exposed form & focus outline in DB results.
          if (!document.getElementById('ai-hide-exposed')) {
            const style = document.createElement('style');
            style.id = 'ai-hide-exposed';
            style.textContent = `
              #ai-search-block-db-results .views-exposed-form,
              #ai-db-results-block .views-exposed-form,
              .ai-db-results .views-exposed-form,
              #ai-search-block-db-results .view-empty,
              #ai-db-results-block .view-empty,
              .ai-db-results .view-empty { display: none !important; }
              #ai-search-block-db-results:focus,
              #ai-db-results-block:focus,
              .ai-db-results:focus { outline: 0 !important; box-shadow: none !important; }
              `;
            document.head.appendChild(style);
          }

          // Handle Enter key submission for textarea if enabled.
          const inputField = drupalSettings.ai_search_block?.input_field;
          const submitOnEnter = drupalSettings.ai_search_block?.submit_on_enter;

          if (inputField === 'textarea' && submitOnEnter) {
            const $queryField = $form.find(
              '[data-drupal-selector="edit-query"]',
            );
            $queryField.on('keydown', function (e) {
              // Check if Enter was pressed without Shift
              if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                $form.trigger('submit');
              }
            });
          }

          // Hide suffix text initially.
          if ($suffixText.length) {
            $suffixText.hide();
          }

          // Determine loading message and create a reusable loader element.
          const loadingMsg =
            drupalSettings.ai_search_block &&
            drupalSettings.ai_search_block.loading_text
              ? drupalSettings.ai_search_block.loading_text
              : 'Loading...';
          const $loader = $(
            `<p class="loading_text"><span class="loader"></span>${
              loadingMsg
            }</p>`,
          );

          // --- Helpers for DB results (minimal additions) ---

          function getScrollOffset() {
            let extra = 0;
            const $toolbar = $('#toolbar-bar:visible');
            if ($toolbar.length) {
              extra += $toolbar.outerHeight();
            }
            const $fixedHeader = $(
              '.site-header.is-fixed:visible, header.fixed:visible',
            );
            if ($fixedHeader.length) {
              extra += $fixedHeader.outerHeight();
            }
            const cfg =
              drupalSettings.ai_search_block &&
              drupalSettings.ai_search_block.scroll_offset;
            if (typeof cfg === 'number') {
              extra += cfg;
            }
            return extra;
          }

          function scrollToResults($target) {
            if (!$target || !$target.length) {
              return;
            }
            const top = Math.max(
              0,
              $target.offset().top - getScrollOffset() - 12,
            );
            $('html, body').scrollTop(top);
            $target.attr('tabindex', '-1').trigger('focus');
          }

          function hideExposedForm($target) {
            const $exposed = $target.find('form.views-exposed-form');
            if (!$exposed.length) {
              return;
            }
            $exposed.attr('aria-hidden', 'true').addClass('ai-exposed-hidden');
            $exposed[0].style.display = 'none';
            $exposed
              .find('input,select,textarea,button,a')
              .attr('tabindex', '-1');
          }

          function fixPager($target, currentPage) {
            const $ul = $target
              .find('ul.pagination, ul.js-pager__items')
              .first();
            if (!$ul.length) {
              return;
            }

            $target.data('currentPage', currentPage);

            $ul
              .find('li.page-item')
              .removeClass('active')
              .each(function itemEach() {
                const $li = $(this);
                const $span = $li.find('> span.page-link');
                if ($span.length) {
                  const label = $span.html();
                  $span.replaceWith(
                    $('<a class="page-link" href="#"></a>').html(label),
                  );
                }
              });

            const targetLabel = (currentPage + 1).toString();
            const $liForPage = $ul
              .find('li.page-item')
              .filter(function itemFilter() {
                const $el = $(this)
                  .find('> a.page-link, > span.page-link')
                  .first();
                const text = ($el.html() || '').trim();
                return /^\d+$/.test(text) && text === targetLabel;
              })
              .first();

            if ($liForPage.length) {
              $liForPage.addClass('active');
              const $a = $liForPage.find('> a.page-link');
              if ($a.length) {
                $a.replaceWith(
                  $('<span class="page-link"></span>').html(targetLabel),
                );
              }
            }

            $ul.find('li.page-item').each(function pageItemEach() {
              const $li = $(this);
              const $el = $li.find('> a.page-link, > span.page-link').first();
              const txt = ($el.html() || '').trim();

              if (/^\d+$/.test(txt)) {
                const n = Number.parseInt(txt, 10) - 1;
                $li.attr('data-page', n);
                if ($el[0] && $el[0].tagName === 'A') {
                  $el.attr('href', '#').attr('role', 'button');
                }
              } else {
                $li.removeAttr('data-page');
              }
            });

            $ul.find('li.page-item[data-synth]').remove();
            if (currentPage > 0) {
              const $prevLi = $(
                `<li class="page-item" data-synth="prev">
<a class="page-link" href="#" role="button" title="Go to previous page">
<span aria-hidden="true">‹‹</span><span class="visually-hidden">Previous page</span></a></li>`,
              );

              const $firstLi = $(
                `<li class="page-item" data-synth="first">
<a class="page-link" href="#" role="button" title="Go to first page">
<span aria-hidden="true">« First</span><span class="visually-hidden">First page</span></a></li>`,
              );

              const $firstNumeric = $ul
                .find('li.page-item')
                .filter(function numericFilter() {
                  const t = $(this)
                    .find('> a.page-link, > span.page-link')
                    .first()
                    .html()
                    .trim();
                  return /^\d+$/.test(t);
                })
                .first();

              if ($firstNumeric.length) {
                $firstNumeric.before($prevLi).before($firstLi);
              } else {
                $ul.prepend($firstLi).prepend($prevLi);
              }
            }
          }

          function fetchDbResults(page, queryVal, blockIdVal) {
            const pageNum = typeof page === 'undefined' ? 0 : page;

            const seq =
              (Number.parseInt($dbResults.data('reqSeq'), 10) || 0) + 1;
            $dbResults.data('reqSeq', seq);

            console.log(
              'AI Search: REQUEST → DB results, requested page:',
              pageNum,
              'seq:',
              seq,
            );
            dispatchStatusEvent('page-requested', { page: pageNum, seq });

            if (
              !drupalSettings.ai_search_block ||
              !drupalSettings.ai_search_block.enable_database_results
            ) {
              console.log('AI Search: DB results disabled in settings');
              return;
            }

            $dbResults.html(
              '<p class="loading_text">Loading database results...</p>',
            );

            $.ajax({
              url:
                (drupalSettings.ai_search_block &&
                  drupalSettings.ai_search_block.db_results_url) ||
                '/ai-search-block/db-results',
              type: 'POST',
              data: {
                query: queryVal,
                block_id: blockIdVal,
                page: pageNum,
              },
              success(data) {
                const latestSeq =
                  Number.parseInt($dbResults.data('reqSeq'), 10) || 0;
                if (seq !== latestSeq) {
                  console.warn(
                    `AI Search: STALE response ignored (seq${seq} latest${latestSeq})`,
                  );
                  return;
                }

                console.log(
                  'AI Search: RESPONSE ← DB results, requested page:',
                  pageNum,
                  'seq:',
                  seq,
                );

                if (data && data.html) {
                  $dbResults.html(data.html);

                  // Remove use-ajax to avoid core AJAX handling.
                  $dbResults.find('a.use-ajax').removeClass('use-ajax');

                  // Ensure Views responsive grid CSS is present.
                  if (
                    !$('link[href*="views-responsive-grid.css"]').length &&
                    drupalSettings.path &&
                    drupalSettings.path.baseUrl !== undefined
                  ) {
                    const link = document.createElement('link');
                    link.rel = 'stylesheet';
                    link.href = `${drupalSettings.path.baseUrl}core/modules/views/css/views-responsive-grid.css`;
                    document.head.appendChild(link);
                  }

                  Drupal.attachBehaviors($dbResults[0]);

                  fixPager($dbResults, pageNum);

                  $dbResults.data('currentPage', pageNum);
                  console.log(
                    'AI Search: LOADED/APPLIED page:',
                    $dbResults.data('currentPage'),
                  );

                  hideExposedForm($dbResults);

                  scrollToResults($dbResults);

                  $dbResults
                    .off('click.aiPager')
                    .on('click.aiPager', 'a.page-link', function pagerClick(e) {
                      e.preventDefault();
                      e.stopPropagation();
                      e.stopImmediatePropagation();

                      const $a = $(this);
                      const $li = $a.closest('li.page-item');
                      const cur =
                        Number.parseInt($dbResults.data('currentPage'), 10) ||
                        0;
                      let nextPage;

                      if ($li.attr('data-page')) {
                        nextPage = Number.parseInt($li.attr('data-page'), 10);
                      } else if ($li.attr('data-synth') === 'first') {
                        nextPage = 0;
                      } else if ($li.attr('data-synth') === 'prev') {
                        nextPage = Math.max(0, cur - 1);
                      } else {
                        const label = ($a.html() || '').trim().toLowerCase();
                        if (
                          label.includes('next') ||
                          label === '›' ||
                          label === '››'
                        ) {
                          nextPage = cur + 1;
                        } else if (
                          label.includes('prev') ||
                          label === '‹' ||
                          label === '‹‹'
                        ) {
                          nextPage = Math.max(0, cur - 1);
                        }
                      }

                      if (
                        typeof nextPage === 'number' &&
                        !Number.isNaN(nextPage)
                      ) {
                        console.log(
                          'AI Search: REQUESTED page (via click):',
                          nextPage,
                          'current:',
                          cur,
                        );
                        fetchDbResults(nextPage, queryVal, blockIdVal);
                        scrollToResults($dbResults);
                        return false;
                      }
                      return true;
                    });
                } else {
                  $dbResults.html('<p>No database results found.</p>');
                }
              },
              error(jqXHR, textStatus, errorThrown) {
                const latestSeq =
                  Number.parseInt($dbResults.data('reqSeq'), 10) || 0;
                console.error(
                  'AI Search: DB results ERROR for requested page:',
                  pageNum,
                  'seq:',
                  seq,
                  textStatus,
                  errorThrown,
                  'latest seq:',
                  latestSeq,
                );
                $dbResults.html('<p>Error loading database results.</p>');
              },
            });
          }

          // ---------------- End Helpers ------------------

          $form.on('submit', function (event) {
            console.log('Submit handler triggered', new Date().getTime());
            event.preventDefault();

            const submitButton = $form.find(
              '[data-drupal-selector="edit-submit"]',
            );
            submitButton.prop('disabled', true);

            // Show the loader initially.
            $resultsBlock.html($loader);
            $dbResults.empty();

            // Dispatch loading status
            dispatchStatusEvent('loading', { form: $form[0] });

            // Retrieve form values.
            const queryVal =
              document.querySelector('[data-drupal-selector="edit-query"]')
                .value || '';
            const streamVal =
              document.querySelector('[data-drupal-selector="edit-stream"]')
                .value || '';
            const blockIdVal =
              document.querySelector('[data-drupal-selector="edit-block-id"]')
                .value || '';

            // Clear the form if clear_after_search is enabled.
            const clearAfterSearch =
              drupalSettings.ai_search_block?.clear_after_search;
            if (clearAfterSearch) {
              const queryField = document.querySelector(
                '[data-drupal-selector="edit-query"]',
              );
              if (queryField) {
                queryField.value = '';
              }
            }

            // If streaming is enabled (using '1' as true).
            if (streamVal === 'true') {
              $suffixText.hide();
              try {
                const xhr = new XMLHttpRequest();
                xhr.open(
                  'POST',
                  drupalSettings.ai_search_block.submit_url,
                  true,
                );
                xhr.setRequestHeader('Content-Type', 'application/json');
                xhr.setRequestHeader('Accept', 'application/json');

                // Cache variables to hold the full output.
                let lastResponseLength = 0;
                let joined = '';

                xhr.onprogress = function () {
                  const responseText = xhr.responseText || '';
                  // Get only the new part of the response.
                  const newData = responseText.substring(lastResponseLength);
                  lastResponseLength = responseText.length;

                  // Split new data using the delimiter.
                  const chunks = newData.trim().split('|§|').filter(Boolean);

                  // Parse each chunk and accumulate the answer pieces.
                  chunks.forEach(function (chunk) {
                    try {
                      const parsed = JSON.parse(chunk);
                      // Update log Id from each chunk.
                      if (parsed.log_id) {
                        drupalSettings.ai_search_block.logId = parsed.log_id;
                      }
                      joined += parsed.answer_piece || '';
                    } catch (e) {
                      console.error('Error parsing chunk:', e, chunk);
                    }
                  });

                  // Overwrite the full output (letting browsers fix broken HTML)
                  // and re-append the loader.
                  $resultsBlock.html(joined).append($loader);

                  // Dispatch streaming progress status
                  dispatchStatusEvent('streaming', {
                    progress: joined.length,
                    form: $form[0],
                  });
                };

                xhr.onreadystatechange = function () {
                  if (xhr.readyState === 4) {
                    if (xhr.status === 200) {
                      // Remove the loader upon successful completion.
                      $loader.remove();
                      submitButton.prop('disabled', false);
                      if ($suffixText.length) {
                        $suffixText.html(
                          drupalSettings.ai_search_block.suffix_text,
                        );
                        Drupal.attachBehaviors($suffixText[0]);
                        $suffixText.show();
                      }
                      // Dispatch done status
                      dispatchStatusEvent('done', {
                        response: joined,
                        form: $form[0],
                      });

                      // Fetch DB results after successful streaming response.
                      if (
                        drupalSettings.ai_search_block &&
                        drupalSettings.ai_search_block.enable_database_results
                      ) {
                        console.log(
                          'AI Search: Fetching DB results after stream',
                        );
                        $dbResults.data('currentPage', 0);
                        fetchDbResults(0, queryVal, blockIdVal);
                      }
                    } else if (xhr.status === 500) {
                      $resultsBlock.html('An error happened.');
                      console.error('Error response:', xhr.responseText);
                      submitButton.prop('disabled', false);
                      // Dispatch error status
                      dispatchStatusEvent('error', {
                        error: xhr.responseText,
                        form: $form[0],
                      });
                      try {
                        const parsedError = JSON.parse(xhr.responseText);
                        if (
                          parsedError.response &&
                          parsedError.response.answer_piece
                        ) {
                          $resultsBlock.html(parsedError.response.answer_piece);
                        }
                        Drupal.attachBehaviors($resultsBlock[0]);
                      } catch (e) {
                        console.error('Error parsing 500 response:', e);
                      }
                    }
                  }
                };

                // Send the streaming request.
                xhr.send(
                  JSON.stringify({
                    query: queryVal,
                    stream: streamVal,
                    block_id: blockIdVal,
                  }),
                );
              } catch (e) {
                console.error('XHR error:', e);
                dispatchStatusEvent('error', { error: e, form: $form[0] });
              }
            } else {
              $suffixText.hide();
              // Non-streaming: use jQuery.post.
              $.post(
                drupalSettings.ai_search_block.submit_url,
                {
                  query: queryVal,
                  stream: streamVal,
                  block_id: blockIdVal,
                },
                function (data) {
                  if (data && data.response) {
                    $resultsBlock.html(data.response);
                  }
                  // Set log Id if available.
                  if (data && data.log_id) {
                    drupalSettings.ai_search_block.logId = data.log_id;
                  }
                  if ($suffixText.length) {
                    $suffixText
                      .html(drupalSettings.ai_search_block.suffix_text)
                      .show();
                    Drupal.attachBehaviors($suffixText[0]);
                  }
                  submitButton.prop('disabled', false);
                  dispatchStatusEvent('done', {
                    response: data,
                    form: $form[0],
                  });
                },
              )
                .fail(function (jqXHR) {
                  $resultsBlock.html('An error happened.');
                  console.error('Error on non-streaming request');
                  dispatchStatusEvent('error', {
                    error: jqXHR.responseText,
                    form: $form[0],
                  });
                })
                .always(function () {
                  // Fetch DB results after non-streaming request completes.
                  if (
                    drupalSettings.ai_search_block &&
                    drupalSettings.ai_search_block.enable_database_results
                  ) {
                    console.log('AI Search: Fetching DB results after POST');
                    $dbResults.data('currentPage', 0);
                    fetchDbResults(0, queryVal, blockIdVal);
                  }
                  submitButton.prop('disabled', false);
                });
            }

            return false;
          });
        },
      );
    },
  };
})(jQuery, Drupal, drupalSettings, once);
