<?php

namespace Drupal\ai_seo\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * AI SEO configuration form.
 */
class ConfigurationForm extends ConfigFormBase {

  /**
   * AI analyzer.
   *
   * @var \Drupal\ai_seo\AiSeoAnalyzer
   */
  protected $analyzer;

  /**
   * The provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->analyzer = $container->get('ai_seo.service');
    $instance->providerManager = $container->get('ai.provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'ai_seo.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_seo_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $preferences_token = NULL) {
    // Create the form.
    $config = $this->config('ai_seo.settings');
    // Get models from AI provider.
    $chat_models = $this->providerManager->getSimpleProviderModelOptions('chat');
    $default_chat_model = $this->providerManager->getSimpleDefaultProviderOptions('chat');
    array_shift($chat_models);
    array_splice($chat_models, 0, 1);
    $form['provider_and_model'] = [
      '#type' => 'select',
      '#options' => $chat_models,
      '#disabled' => count($chat_models) == 0,
      "#empty_option" => $this->t('-- Default from AI module (chat) --'),
      '#default_value' => $config->get('provider_and_model') ?? $default_chat_model,
      '#title' => $this->t('Choose Provider and Model used for SEO analyzing.'),
    ];

    $form['prompt'] = [
      '#type' => 'details',
      '#title' => $this->t('Prompts'),
      '#open' => TRUE,
    ];

    $form['prompt']['system'] = [
      '#type' => 'details',
      '#title' => $this->t('System prompt'),
      '#open' => FALSE,
    ];

    $form['prompt']['system']['default_system_prompt'] = [
      '#type' => 'textarea',
      '#readonly' => TRUE,
      '#disabled' => TRUE,
      '#title' => $this->t('Default prompt'),
      '#description' => $this->t('The default system prompt comes with the module and it is the one that is used unless a custom prompt is provided below.'),
      '#value' => $this->analyzer->getDefaultSystemPrompt(),
    ];

    $form['prompt']['system']['custom_system_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System prompt'),
      '#default_value' => $config->get('system_prompt') ?? '',
    ];

    $form['prompt']['default_prompt'] = [
      '#type' => 'textarea',
      '#readonly' => TRUE,
      '#disabled' => TRUE,
      '#title' => $this->t('Default prompt'),
      '#description' => $this->t('The default prompt comes with the module and it is the one that is used unless a custom prompt is provided below.'),
      '#value' => $this->analyzer->getDefaultPrompt(),
    ];

    $form['prompt']['custom_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom prompt'),
      '#default_value' => $config->get('custom_prompt') ?? '',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ?string $preferences_token = NULL) {
    $config = $this->config('ai_seo.settings');
    $custom_system_prompt = $form_state->getValue('custom_system_prompt') ?? '';
    $custom_prompt = $form_state->getValue('custom_prompt') ?? '';
    $config
      ->set('custom_system_prompt', trim($custom_system_prompt))
      ->set('custom_prompt', trim($custom_prompt))
      ->set('provider_and_model', $form_state->getValue('provider_and_model'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Page title.
   */
  public function getTitle() {
    return $this->t('Administer AI SEO analyzer settings');
  }

}
