<?php

namespace Drupal\ai_seo\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\ai_seo\ReportService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class ViewReportController.
 */
class ViewReportController extends ControllerBase {

  /**
   * The report service.
   *
   * @var \Drupal\ai_seo\ReportService
   */
  protected $reportService;

  /**
   * Constructs a ViewReportController object.
   *
   * @param \Drupal\ai_seo\ReportService $report_service
   *   The report service.
   */
  public function __construct(ReportService $report_service) {
    $this->reportService = $report_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_seo.report_service')
    );
  }

  /**
   * Build the page to display a specific report.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param int $report_id
   *   The report ID/index.
   *
   * @return array
   *   Render array for the report display.
   */
  public function printReport(\Drupal\node\NodeInterface $node, $report_id) {
    // Convert report_id to integer
    $report_id = (int) $report_id;

    // Get the specific report
    $report = $this->reportService->getFormattedReport($node->id(), $report_id);

    // Check if the requested report exists
    if (!$report) {
      throw new NotFoundHttpException('Report not found.');
    }

    // Get report number for display
    $report_number = $this->reportService->getReportNumber($node->id(), $report_id);


    $build = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['seo-report-view'],
      ],
    ];

    // Back link to main SEO page
    $build['back_link'] = [
      '#type' => 'link',
      '#title' => $this->t('← Back to SEO Analysis'),
      '#url' => \Drupal\Core\Url::fromRoute('entity.node.seo_analyzer', ['node' => $node->id()]),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#weight' => -10,
    ];

    // Report metadata
    $build['metadata'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['report-metadata'],
        'style' => 'margin: 20px 0; padding: 15px; background: #f8f9fa; border-left: 4px solid #007cba;',
      ],
      'info' => [
        '#markup' => '
        <div><strong>Report #' . $report_number . '</strong></div>
        <div><strong>Report Type:</strong> ' . $report['report_type'] . '</div>
        <div><strong>Created:</strong> ' . $report['time_ago'] . '</div>
        <div><strong>Node:</strong> ' . $node->getTitle() . '</div>' .
        $report['revision_row'],
      ],
    ];

    // Main report content
    $build['report_content'] = [
      '#type' => 'details',
      '#title' => $this->t('SEO Analysis Report'),
      '#open' => TRUE,
      '#attributes' => [
        'class' => ['report-content'],
        'style' => 'max-width: 1200px; margin: 20px 0;',
      ],
      'content' => [
        '#markup' => $report['report'],
      ],
    ];

    // Prompt used (collapsible)
    $build['prompt_details'] = [
      '#type' => 'details',
      '#title' => $this->t('Prompt Used'),
      '#open' => FALSE,
      '#attributes' => [
        'class' => ['report-prompt'],
        'style' => 'max-width: 1200px; margin: 20px 0;',
      ],
      'prompt' => [
        '#type' => 'textarea',
        '#title' => $this->t('AI Prompt'),
        '#value' => $report['prompt'],
        '#rows' => 10,
        '#attributes' => [
          'readonly' => 'readonly',
        ],
      ],
    ];

    // HTML analyzed (collapsible)
    $build['html_details'] = [
      '#type' => 'details',
      '#title' => $this->t('HTML Analyzed'),
      '#open' => FALSE,
      '#attributes' => [
        'class' => ['report-html'],
        'style' => 'max-width: 1200px; margin: 20px 0;',
      ],
      'html' => [
        '#type' => 'textarea',
        '#title' => $this->t('Source HTML'),
        '#value' => $report['html'],
        '#rows' => 15,
        '#attributes' => [
          'readonly' => 'readonly',
        ],
      ],
    ];

    // Add CSS for better styling
    $build['#attached']['library'][] = 'ai_seo/report_styles';

    return $build;
  }

  /**
   * Page title callback.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param int $report_id
   *   The report ID.
   *
   * @return string
   *   The page title.
   */
  public function getTitle(\Drupal\node\NodeInterface $node = NULL, $report_id = NULL) {
    if ($node && $report_id !== NULL) {
      $report_id = (int) $report_id;
      $formatted_reports = $this->reportService->getFormattedReports($node->id());
      if (isset($formatted_reports[$report_id])) {
        $report_number = count($formatted_reports) - $report_id;
        return $this->t('SEO Report #@number for "@title"', [
          '@number' => $report_number,
          '@title' => $node->getTitle(),
        ]);
      }
    }
    return $this->t('SEO Report');
  }

}
