<?php

namespace Drupal\ai_seo_link_advisor\Controller;

use Drupal\ai_seo_link_advisor\Analyzer;
use Drupal\ai_seo_link_advisor\Service\DynamicAiServiceLoader;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns response for SEO.
 */
class SeolinkcheckController extends ControllerBase {

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The Dynamic AI Service Loader.
   *
   * @var \Drupal\ai_seo_link_advisor\Service\DynamicAiServiceLoader
   */
  protected $dynamicAiServiceLoader;

  /**
   * The Form Builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The Logger Channel Factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Module Handler.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The Language Manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Background classes for status colors.
   *
   * @var array
   */
  protected $backgroundClasses;

  /**
   * {@inheritdoc}
   */
  public function __construct(Request $request, DynamicAiServiceLoader $dynamicAiServiceLoader, FormBuilderInterface $formBuilder, LoggerChannelFactoryInterface $loggerFactory, ConfigFactoryInterface $configFactory, ModuleExtensionList $moduleExtensionList, LanguageManagerInterface $languageManager) {
    $this->request = $request;
    $this->dynamicAiServiceLoader = $dynamicAiServiceLoader;
    $this->formBuilder = $formBuilder;
    $this->loggerFactory = $loggerFactory;
    $this->configFactory = $configFactory;
    $this->moduleExtensionList = $moduleExtensionList;
    $this->languageManager = $languageManager;
    $this->backgroundClasses = [
      0 => 'bg-success',
      1 => 'bg-warning',
      2 => 'bg-warning',
      3 => 'bg-warning',
      4 => 'bg-warning',
      5 => 'bg-warning',
      6 => 'bg-danger',
      7 => 'bg-danger',
      8 => 'bg-danger',
      9 => 'bg-danger',
      10 => 'bg-danger',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('ai_seo_link_advisor.dynamic_ai_service_loader'),
      $container->get('form_builder'),
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('extension.list.module'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function generateAnalyzerPage() {
    // Retrieve the 'url' query parameter from the request.
    $url = $this->request->query->get('url') ?? '';

    // Sanitize url.
    $url = $this->sanitizeUrl($url);

    // Get the configuration for the AI SEO Link Advisor module.
    $config = $this->configFactory->get('ai_seo_link_advisor.settings');
    $ai_service = $config->get('ai_service');

    // Initialize the output array for rendering the page.
    $output = [
      '#theme' => 'page_ai_seo_link_advisor',
      // Add the Seourl form to the page.
      '#search_form' => $this->formBuilder->getForm('Drupal\ai_seo_link_advisor\Form\SeourlForm'),
      // Attach the required library for styling.
      '#attached' => ['library' => ['ai_seo_link_advisor/styling']],
    ];

    // If the URL is not provided, return the basic output with the search form.
    if (empty($url)) {
      return $output;
    }

    // Initialize arrays to hold analysis results and SEO recommendations.
    $results = [];
    $recommendations = [];
    try {
      // Get the AI service dynamically.
      $ai_service_instance = $this->dynamicAiServiceLoader->getAiService();
      if ($ai_service_instance) {
        // Analyze the URL and get SEO results.
        $results = (new Analyzer())->analyzeUrl($url, '', $this->languageManager->getCurrentLanguage()->getId());
        $text = 'Give me SEO recommendations for the URL: ' . $url;

        if ($ai_service === 'openai') {
          $recommendations = $ai_service_instance->getSeoRecommendations($url);
        }
        elseif ($ai_service === 'gemini') {
          $recommendations = $ai_service_instance->generateContent($text);
        }
        // Add detailed content analytics and general analytics to the output.
        $output['#content_analytics'] = $this->createContentAnalyticsForm($results, $url, $url, $ai_service);
        $output['#general_analytics'] = $this->createGeneralAnalyticsForm($results, $url, $url);
        $output['#recommendations'] = $recommendations;
      }
      else {
        // Generate a link to setting page if the AI service is not configured.
        $config_url = Url::fromRoute('ai_seo_link_advisor.settings')->toString();
        $output['#error_message'] = $this->t('Kindly enter the API key after selecting an AI service. <a href="@config_url">Click here</a> to configure the AI service.', ['@config_url' => $config_url]);
      }
    }
    catch (\Exception $e) {
      // Log any errors encountered during URL analysis.
      $this->loggerFactory->get('ai_seo_link_advisor')->error('Error analyzing URL: @url. Error: @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);

      // Provide a user-friendly error message if analysis fails.
      $output['#error_message'] = $this->t('The URL could not be analyzed. Please verify that the URL is correct and try again.');
    }
    // Return the constructed output array for rendering.
    return $output;
  }

  /**
   * Sanitizes and validates a user-provided URL.
   *
   * This function performs the following checks:
   * - Trims the URL and checks if it's a valid format.
   * - Ensures the scheme is either HTTP or HTTPS.
   * - Allow only trusted hosts / URLs that belong to the current domain.
   * - Ensures the URL maps to a valid route in the Drupal site.
   * - Checks if the current user has access to the resolved route.
   * - Disallows URLs that point to administrative pages.
   *
   * @param string $url
   *   The input URL string to sanitize.
   *
   * @return string
   *   The sanitized URL if valid and secure, or an empty string otherwise.
   */
  public static function sanitizeUrl(string $url): string {
    // Proceed only if the URL is not empty.
    if (!empty($url)) {
      // Remove leading/trailing whitespace.
      $url = trim($url);
      // Break down the URL into components.
      $parsed_url = parse_url($url);

      // Validate the URL format.
      if (!filter_var($url, FILTER_VALIDATE_URL)) {
        // URL is not valid.
        \Drupal::messenger()->addError(t('The URL is not valid.'));
        return '';
      }

      // Reject schemes other than http or https.
      if (
        !empty($parsed_url['scheme']) &&
        !in_array($parsed_url['scheme'], ['http', 'https'])
      ) {
        \Drupal::messenger()->addError(t('Only HTTP and HTTPS URLs are allowed.'));
        return '';
      }

      // Allow only trusted hosts.
      $trusted_host_patterns = \Drupal::service('settings')->get('trusted_host_patterns') ?? [];
      if (!empty($trusted_host_patterns)) {
        $valid = FALSE;
        foreach ($trusted_host_patterns as $pattern) {
          if (preg_match('#' . $pattern . '#', $parsed_url['host'])) {
            $valid = TRUE;
            break;
          }
        }
        if (!$valid) {
          \Drupal::messenger()->addError(t('Only URLs from the trusted hosts are allowed.'));
          return '';
        }
      }
      else {
        \Drupal::messenger()->addError(t('Please configure the trusted_host_patterns in your settings.php file to ensure proper host validation.'));
        return '';
      }

      // Default to '/' if a path is not set.
      $path = $parsed_url['path'] ?? '/';
      // Create a relative URL object from the path.
      $url_object = Url::fromUserInput($path, ['absolute' => FALSE]);

      // Ensure the path maps to a valid routed URL.
      if (!$url_object->isRouted()) {
        \Drupal::messenger()->addError(t('The URL does not resolve to a Drupal page.'));
        return '';
      }

      // Verify if the user has access to the resolved route.
      if (!$url_object->access()) {
        \Drupal::messenger()->addError(t('You do not have access to the resolved route.'));
        return '';
      }

      // Disallow URLs pointing to administrative routes.
      $route = \Drupal::service('router.route_provider')->getRouteByName($url_object->getRouteName());
      if ($route->getOption('_admin_route') || str_starts_with($path, '/admin')) {
        \Drupal::messenger()->addError(t('You cannot analyze administrative pages.'));
        return '';
      }
    }

    return $url;
  }

  /**
   * Creates a form for displaying content analytics based on the SEO results.
   *
   * @param array $results
   *   The SEO analysis results.
   * @param string $url
   *   The URL being analyzed.
   * @param string $page_url
   *   The page URL for display purposes.
   * @param string $ai_service
   *   The AI service used for recommendations.
   *
   * @return array
   *   A renderable array for content analytics.
   */
  private function createContentAnalyticsForm($results, $url, $page_url, $ai_service) {
    // Initialize the content analytics form array.
    $content_analytics_form = [];
    $content_analytics_form['content_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("Content analytics for @url", ['@url' => $url]),
      '#open' => FALSE,
    ];
    // Prepare data for the content analytics table.
    $data = [
      'meta_tag_length' => $results['PageMeta'],
      'heading_structure' => $results['PageHeaders'],
      'page_content_ratio' => $results['PageContentRatio'],
      'image_alt_texts' => $results['PageAlts'],
      'page_url_size' => $results['PageUrlLength'],
    ];
    // Add AI-specific data if using Gemini or OpenAI services.
    if ($ai_service === 'gemini') {
      $data['ai_services'] = 'gemini';
      $data['meta_tag_length_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description']);

      $data['heading_structure_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value']);

      $data['page_content_ratio_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value']);

      $data['image_alt_texts_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt']);

      $data['page_url_size_ai'] = $this->dynamicAiServiceLoader->getAiService()->generateContent($results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value']);
    }
    elseif ($ai_service === 'openai') {
      $data['ai_services'] = 'openai';
      $data['meta_tag_length_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description']);

      $data['heading_structure_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value']);

      $data['page_content_ratio_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value']);

      $data['image_alt_texts_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt']);

      $data['page_url_size_ai'] = $this->dynamicAiServiceLoader->getAiService()->getSeoRecommendations($results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value']);
    }
    // Get the module path for images.
    $module_path = $this->moduleExtensionList->getPath('ai_seo_link_advisor');
    $base_path = base_path() . $module_path . '/icons/';
    // Create the content analytics form with the data and icons.
    $content_analytics_form['content_analytics']['content'] = [
      '#theme' => 'table_content_analytics',
      '#page_url' => $page_url,
      '#data' => $data,
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];

    return $content_analytics_form;
  }

  /**
   * Creates a form for displaying general analytics based on the SEO results.
   *
   * @param array $results
   *   The SEO analysis results.
   * @param string $url
   *   The URL being analyzed.
   * @param string $page_url
   *   The page URL for display purposes.
   *
   * @return array
   *   A renderable array for general analytics.
   */
  private function createGeneralAnalyticsForm($results, $url, $page_url) {
    // Initialize the general analytics form array.
    $general_analytics_form = [];
    $general_analytics_form['general_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("General analytics for @url", ['@url' => $url]),
      '#open' => FALSE,
    ];

    // Get the module path for images.
    $module_path = $this->moduleExtensionList->getPath('ai_seo_link_advisor');
    $base_path = base_path() . $module_path . '/icons/';

    // Create the general analytics form with the data and icons.
    $general_analytics_form['general_analytics']['content'] = [
      '#theme' => 'table_general_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => [
        'is_encrypted' => $results['PageSSL'],
        'has_redirect' => $results['PageRedirect'],
        'content_size' => $results['PageContentSize'],
        'page_load_time' => $results['PageLoadTime'],
        'robots_txt' => $results['FileRobots'],
        'sitemap' => $results['FileSitemap'],
      ],
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];

    return $general_analytics_form;
  }

  /**
   * Creates a form to display SEO recommendations.
   *
   * This method constructs a form array that includes SEO recommendations.
   *
   * @param array $recommendations
   *   An associative array of SEO recommendations to be displayed in the form.
   *
   * @return array
   *   A renderable array representing the form structure.
   */
  private function createRecommendationsForm($recommendations) {
    // Initialize the form array.
    $form = [];
    // Define a 'details' element to wrap the SEO recommendations.
    $form['recommendations'] = [
      '#type' => 'details',
      '#title' => $this->t("SEO Recommendations"),
      '#open' => TRUE,
    ];
    // Add the actual content of the recommendations inside the details element.
    $form['recommendations']['content'] = [
      '#theme' => 'page_ai_seo_link_advisor',
      '#data' => $recommendations,
    ];
    return $form;
  }

}
