<?php

declare(strict_types=1);

namespace Drupal\ai_seo_link_advisor;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides SEO analysis and AI-assisted recommendations for a given URL.
 */
final class AiSeoLinkAdvisor {

  use StringTranslationTrait;

  /**
   * Background classes for status colors.
   *
   * @var array
   */
  protected array $backgroundClasses;

  public function __construct(
    private readonly AiProviderPluginManager $aiProvider,
    private readonly RequestStack $requestStack,
    private readonly LoggerChannelFactoryInterface $loggerFactory,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly ModuleExtensionList $moduleExtensionList,
    private readonly LanguageManagerInterface $languageManager,
  ) {
    $this->backgroundClasses = [
      0 => 'bg-success',
      1 => 'bg-warning',
      2 => 'bg-warning',
      3 => 'bg-warning',
      4 => 'bg-warning',
      5 => 'bg-warning',
      6 => 'bg-danger',
      7 => 'bg-danger',
      8 => 'bg-danger',
      9 => 'bg-danger',
      10 => 'bg-danger',
    ];
  }

  /**
   * Analyzes a given URL and prepares data for SEO analysis rendering.
   */
  public function analyze($url): array {
    // Initialize the output array for rendering the page.
    $output = [
      '#theme' => 'page_ai_seo_link_advisor',
    ];

    try {
      // Analyze the URL and get SEO results.
      $results = (new Analyzer())->analyzeUrl($url, '', $this->languageManager->getCurrentLanguage()->getId()) ?? [];

      // Add detailed content analytics and general analytics to the output.
      $output['#content_analytics'] = $this->createContentAnalyticsForm($results, $url, $url);
      $output['#general_analytics'] = $this->createGeneralAnalyticsForm($results, $url, $url);
    }
    catch (\Exception $e) {
      // Log any errors encountered during URL analysis.
      $this->loggerFactory->get('ai_seo_link_advisor')->error('Error analyzing URL: @url. Error: @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);

      // Provide a user-friendly error message if analysis fails.
      $output['#error_message'] = $this->t('The URL could not be analyzed. Please verify that the URL is correct and try again.');
    }

    // Return the constructed output array for rendering.
    return $output;
  }

  /**
   * Creates a form for displaying content analytics based on the SEO results.
   *
   * @param array $results
   *   The SEO analysis results.
   * @param string $url
   *   The URL being analyzed.
   * @param string $page_url
   *   The page URL for display purposes.
   *
   * @return array
   *   A renderable array for content analytics.
   */
  private function createContentAnalyticsForm(array $results, string $url, string $page_url): array {
    // Initialize the content analytics form an array.
    $content_analytics_form = [];
    $content_analytics_form['content_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("Content analytics for @url", ['@url' => $url]),
      '#open' => FALSE,
    ];

    // Prepare data for the content analytics table.
    $data = [
      'meta_tag_length' => $results['PageMeta'],
      'heading_structure' => $results['PageHeaders'],
      'page_content_ratio' => $results['PageContentRatio'],
      'image_alt_texts' => $results['PageAlts'],
      'page_url_size' => $results['PageUrlLength'],
      'meta_tag_length_ai' => '',
      'heading_structure_ai' => '',
      'page_content_ratio_ai' => '',
      'image_alt_texts_ai' => '',
      'page_url_size_ai' => '',
    ];

    try {
      // Get the AI configuration.
      $model_and_provider_string = $this->aiProvider->getSimpleDefaultProviderOptions('chat');
      // Get provider and model.
      $ai_settings = explode('__', $model_and_provider_string);

      if (count($ai_settings) == 2) {
        $data['ai_services'] = $ai_settings;

        // Chat it up.
        $configuration = [
          'http_client_options' => [
            'timeout' => 500,
          ],
        ];
        $ai_provider = $this->aiProvider->createInstance($ai_settings[0], $configuration);

        if ($data['meta_tag_length']['negative_impact'] != 0) {
          $prompt = $results['PageMeta']['analysis'] . '-' . $results['PageMeta']['value']['meta']['description'];
          $data['meta_tag_length_ai'] = $this->generateAiRecommendations($ai_provider, $ai_settings, $prompt);
        }

        if ($data['heading_structure']['negative_impact'] != 0) {
          $prompt = $results['PageHeaders']['analysis'] . '-' . $results['PageHeaders']['value'];
          $data['heading_structure_ai'] = $this->generateAiRecommendations($ai_provider, $ai_settings, $prompt);
        }

        if ($data['page_content_ratio']['negative_impact'] != 0) {
          $prompt = $results['PageContentRatio']['analysis'] . '-' . $results['PageContentRatio']['value'];
          $data['page_content_ratio_ai'] = $this->generateAiRecommendations($ai_provider, $ai_settings, $prompt);
        }

        if ($data['image_alt_texts']['negative_impact'] != 0) {
          $prompt = $results['PageAlts']['analysis'] . '-' . $results['PageAlts']['value']['images_without_alt'];
          $data['image_alt_texts_ai'] = $this->generateAiRecommendations($ai_provider, $ai_settings, $prompt);
        }

        if ($data['page_url_size']['negative_impact'] != 0) {
          $prompt = $results['PageUrlLength']['analysis'] . '-' . $results['PageUrlLength']['value'];
          $data['page_url_size_ai'] = $this->generateAiRecommendations($ai_provider, $ai_settings, $prompt);
        }
      }
    }
    catch (\Exception $e) {
      // Log any errors encountered during URL analysis.
      $this->loggerFactory->get('ai_seo_link_advisor')->error('Error generating SEO Recommendation using AI for URL: @url. Error: @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);
    }

    // Get the module path for images.
    $module_path = $this->moduleExtensionList->getPath('ai_seo_link_advisor');
    $base_path = base_path() . $module_path . '/icons/';
    // Create the content analytics form with the data and icons.
    $content_analytics_form['content_analytics']['content'] = [
      '#theme' => 'table_content_analytics',
      '#page_url' => $page_url,
      '#data' => $data,
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];
    return $content_analytics_form;
  }

  /**
   * Creates a form for displaying general analytics based on the SEO results.
   *
   * @param array $results
   *   The SEO analysis results.
   * @param string $url
   *   The URL being analyzed.
   * @param string $page_url
   *   The page URL for display purposes.
   *
   * @return array
   *   A renderable array for general analytics.
   */
  private function createGeneralAnalyticsForm(array $results, string $url, string $page_url): array {
    // Initialize the general analytics form array.
    $general_analytics_form = [];
    $general_analytics_form['general_analytics'] = [
      '#type' => 'details',
      '#title' => $this->t("General analytics for @url", ['@url' => $url]),
      '#open' => FALSE,
    ];

    // Get the module path for images.
    $module_path = $this->moduleExtensionList->getPath('ai_seo_link_advisor');
    $base_path = base_path() . $module_path . '/icons/';

    // Create the general analytics form with the data and icons.
    $general_analytics_form['general_analytics']['content'] = [
      '#theme' => 'table_general_analytics',
      '#bgclasses' => $this->backgroundClasses,
      '#page_url' => $page_url,
      '#data' => [
        'is_encrypted' => $results['PageSSL'],
        'has_redirect' => $results['PageRedirect'],
        'content_size' => $results['PageContentSize'],
        'page_load_time' => $results['PageLoadTime'],
        'robots_txt' => $results['FileRobots'],
        'sitemap' => $results['FileSitemap'],
      ],
      '#no_changes_necessary_image' => $base_path . 'checked.png',
      '#attention_needed_image' => $base_path . 'warning.png',
      '#immediate_action_required_image' => $base_path . 'action.png',
    ];
    return $general_analytics_form;
  }

  /**
   * Returns the default system prompt.
   *
   * @return string
   *   The default system prompt.
   */
  private function getDefaultSystemPrompt(): string {
    return "You are an SEO analysis expert specialized in evaluating HTML content from an SEO perspective. You should aim to be thorough, precise, and provide examples wherever possible to illustrate your points. Your role is to provide a solution/recommendation to the following problems:.";
  }

  /**
   * Generates AI-powered recommendations using the configured provider.
   */
  private function generateAiRecommendations($ai_provider, $ai_settings, $prompt): string {
    // Create the chat array to pass on.
    $chat_array = [];
    $chat_array[] = new chatMessage('system', $this->getDefaultSystemPrompt());
    $chat_array[] = new chatMessage('user', $prompt);

    // Create the input chain.
    $messages = new ChatInput($chat_array);
    $message = $ai_provider->chat($messages, $ai_settings[1])->getNormalized();
    $result = trim($message->getText());
    return !empty($result) ? $result : '';
  }

}
