<?php

namespace Drupal\ai_seo_link_advisor;

use Drupal\ai_seo_link_advisor\Analyzer\HttpClient\Client;
use Drupal\ai_seo_link_advisor\Analyzer\HttpClient\ClientInterface;
use Drupal\ai_seo_link_advisor\Analyzer\HttpClient\Exception\HttpException;
use Drupal\ai_seo_link_advisor\Analyzer\Metric\MetricFactory;
use Drupal\ai_seo_link_advisor\Analyzer\Metric\MetricInterface;
use Drupal\Component\Utility\Html;

/**
 * Provides an Analyzer class for SEO analysis.
 */
class Analyzer {

  /**
   * Web page to analyze.
   *
   * @var \Drupal\ai_seo_link_advisor\Page
   */
  public $page;

  /**
   * Default language code to use for translations.
   *
   * @var string
   */
  public $langcode = 'en';

  /**
   * Metrics array.
   *
   * @var array
   */
  public $metrics = [];

  /**
   * HTTP client for making requests.
   *
   * @var \Drupal\ai_seo_link_advisor\Analyzer\HttpClient\ClientInterface
   */
  public $client;

  /**
   * Constructs a new instance of Analyzer.
   *
   * @param \Drupal\ai_seo_link_advisor\Page|null $page
   *   The page to analyze.
   * @param \Drupal\ai_seo_link_advisor\Analyzer\HttpClient\ClientInterface|null $client
   *   The HTTP client for making requests.
   */
  public function __construct(Page|null $page = NULL, ClientInterface|null $client = NULL) {
    $this->client = $client;

    if (empty($client)) {
      $this->client = new Client();
    }

    if (!empty($page)) {
      $this->page = $page;
    }
  }

  /**
   * Analyzes page at specified url.
   */
  public function analyzeUrl(string $url, string|null $keyword = NULL, string|null $langcode = NULL): array {
    if (!empty($langcode)) {
      $this->langcode = $langcode;
    }

    $this->page = new Page($url, $langcode, $this->client);

    if (!empty($keyword)) {
      $this->page->keyword = $keyword;
    }

    return $this->analyze();
  }

  /**
   * Analyzes HTML document from file.
   */
  public function analyzeFile(string $filename, string|null $langcode = NULL): array {
    $this->page = new Page(NULL, $langcode, $this->client);
    $this->page->content = file_get_contents($filename);
    return $this->analyze();
  }

  /**
   * Analyzes an HTML document from string.
   */
  public function analyzeHtml(string $htmlString, string|null $langcode = NULL): array {
    $this->page = new Page(NULL, $langcode, $this->client);
    $this->page->content = $htmlString;
    return $this->analyze();
  }

  /**
   * Starts analysis of a Page.
   */
  public function analyze(): array {
    if (empty($this->metrics)) {
      $this->metrics = $this->getMetrics();
    }

    $results = [];

    foreach ($this->metrics as $metric) {
      if ($metric->name == "FileRobots" || $metric->name == "FileSitemap") {
        $metric->value = Html::escape($metric->value);
      }

      if ($analysisResult = $metric->analyze()) {
        $results[$metric->name] = $this->formatResults($metric, $analysisResult);
      }
    }

    return $results;
  }

  /**
   * Returns available metrics list for a Page.
   */
  public function getMetrics(): array {
    return array_merge($this->page->getMetrics(), $this->getFilesMetrics());
  }

  /**
   * Returns file related metrics.
   */
  public function getFilesMetrics(): array {
    return [
      'robots' => MetricFactory::get('file.robots', $this->getFileContent(
        $this->page->getFactor('url.parsed.scheme') . '://' . $this->page->getFactor('url.parsed.host'),
        'robots.txt'
      )),
      'sitemap' => MetricFactory::get('file.sitemap', $this->getFileContent(
        $this->page->getFactor('url.parsed.scheme') . '://' . $this->page->getFactor('url.parsed.host'),
        'sitemap.xml'
      )),
    ];
  }

  /**
   * Downloads file from the Page's host.
   */
  protected function getFileContent($url, $filename) {
    try {
      $response = $this->client->get($url . '/' . $filename);
      $content = $response->getBody()->getContents();
    }
    catch (HttpException $e) {
      return FALSE;
    }
    return $content;
  }

  /**
   * Formats metric analysis results.
   */
  protected function formatResults(MetricInterface $metric, string $results): array {
    return [
      'analysis' => $results,
      'name' => $metric->name,
      'description' => $metric->description,
      'value' => $metric->value,
      'negative_impact' => $metric->impact,
    ];
  }

}
