<?php

namespace Drupal\ai_seo_link_advisor\Analyzer\Parser;

/**
 * Provides an abstract parser class for SEO analysis.
 */
abstract class AbstractParser implements ParserInterface {
  /**
   * The DOM representation of the HTML document.
   *
   * @var \DOMDocument
   */
  protected $dom;

  /**
   * Constructs an AbstractParser object.
   *
   * @param string|null $html
   *   The HTML document to parse.
   */
  public function __construct(string|null $html = NULL) {
    $this->dom = new \DOMDocument();

    if (!empty($html)) {
      $this->setContent($html);
    }
  }

  /**
   * Sets the content to be analyzed.
   *
   * @param string $html
   *   The HTML document to parse.
   */
  public function setContent(string $html): void {
    $internalErrors = libxml_use_internal_errors(TRUE);
    if (!empty($html)) {
      $this->dom->loadHTML($html, LIBXML_NOWARNING);
    }
    libxml_use_internal_errors($internalErrors);
  }

  /**
   * Removes specified tags along with their content from the DOM.
   *
   * @param string $tag
   *   The tag name to remove.
   */
  protected function removeTags(string $tag): void {
    $tagsToRemove = [];

    foreach ($this->getDomElements($tag) as $element) {
      $tagsToRemove[] = $element;
    }

    foreach ($tagsToRemove as $item) {
      $item->parentNode->removeChild($item);
    }
  }

  /**
   * Returns DOM elements by tag name.
   *
   * @param string $name
   *   The name of the tag to retrieve.
   *
   * @return \DOMNodeList|\DOMElement[]
   *   A list of DOM elements matching the tag name.
   */
  protected function getDomElements(string $name): \DOMNodeList {
    return $this->dom->getElementsByTagName($name);
  }

}
