<?php

namespace Drupal\ai_seo_link_advisor\Analyzer\Parser;

/**
 * Provides a Parser class for SEO analysis.
 */
class Parser extends AbstractParser {

  /**
   * Extracts meta tag information from the DOM.
   */
  public function getMeta(): array {
    $meta = [];

    foreach ($this->getDomElements('meta') as $item) {
      $meta[$item->getAttribute('name')] = trim($item->getAttribute('content'));
    }

    return $meta;
  }

  /**
   * Retrieves all heading (H1–H5) text from the DOM.
   */
  public function getHeaders($keyword = FALSE): array {
    $headers = [];

    for ($x = 1; $x <= 5; $x++) {
      foreach ($this->getDomElements('h' . $x) as $item) {
        $header_text = trim($item->nodeValue);

        if ($keyword && !empty($keyword) && !str_contains($header_text, 'strong')) {
          $header_text = str_ireplace($keyword, "<strong>$keyword</strong>", $header_text);
        }

        $headers['h' . $x][] = $header_text;
      }
    }

    return $headers;
  }

  /**
   * Retrieves the text content of the <title> element from the DOM.
   */
  public function getTitle(): string {
    if ($this->getDomElements('title')->length > 0) {
      return trim($this->getDomElements('title')->item(0)->nodeValue);
    }

    return '';
  }

  /**
   * Extracts image details and alt-text statistics from the DOM.
   */
  public function getImages(): array {
    $data = [
      'total_images' => 0,
      'images_with_alt' => 0,
      'images_without_alt' => 0,
      'images' => [],
    ];
    if ($this->getDomElements('img')->length > 0) {
      foreach ($this->getDomElements('img') as $img) {
        $data['total_images']++;
        $alt = trim($img->getAttribute('alt'));
        if (!empty($alt)) {
          $data['images_with_alt']++;
        }
        if (empty($alt)) {
          $data['images_without_alt']++;
          $alt = 'MISSING!';
        }
        $data['images'][] = [
          'src' => trim($img->getAttribute('src')),
          'alt' => $alt,
          'title' => trim($img->getAttribute('title')),
          'loading' => trim($img->getAttribute('loading')),
          'width' => trim($img->getAttribute('width')),
          'height' => trim($img->getAttribute('height')),
        ];
      }
    }
    return $data;
  }

  /**
   * Extracts and returns cleaned text content from the DOM.
   */
  public function getText(): string {
    $this->removeTags('script');
    $this->removeTags('style');
    $text = strip_tags($this->dom->saveHTML());
    return preg_replace('!\s+!', ' ', strip_tags($text));
  }

}
