<?php

declare(strict_types=1);

namespace Drupal\ai_seo_link_advisor\Form;

use Drupal\ai_seo_link_advisor\AiSeoLinkAdvisor;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a AI SEO Link Advisor form.
 */
final class AiSeoLinkAdvisorForm extends FormBase {

  /**
   * Constructs a new AiSeoLinkAdvisorForm object.
   *
   * @param \Drupal\ai_seo_link_advisor\AiSeoLinkAdvisor $aiSeoLinkAdvisor
   *   The AI SEO Link Advisor service.
   * @param \Drupal\Core\Site\Settings $settings
   *   The settings service.
   * @param \Drupal\Core\Routing\RouteProviderInterface $routeProvider
   *   The route provider service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The route provider service.
   */
  public function __construct(
    protected AiSeoLinkAdvisor $aiSeoLinkAdvisor,
    protected Settings $settings,
    protected RouteProviderInterface $routeProvider,
    protected RendererInterface $renderer,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_seo_link_advisor'),
      $container->get('settings'),
      $container->get('router.route_provider'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_seo_link_advisor_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Define a text field for entering the website URL.
    $form['url'] = [
      '#type' => 'url',
      '#title' => $this->t('URL'),
      '#required' => TRUE,
    ];

    $trusted_host_patterns = $this->settings->get('trusted_host_patterns') ?? [];
    if (empty($trusted_host_patterns)) {
      $this->messenger->addError($this->t('Please configure the trusted_host_patterns in your settings.php file to proceed.'));
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('SEO Analyze'),
      '#disabled' => empty($trusted_host_patterns),
      '#attributes' => ['class' => ['button--primary']],
      '#ajax' => [
        'callback' => '::ajaxCallback',
        'wrapper' => 'markup-wrapper',
        'effect' => 'fade',
      ],
    ];

    $form['output'] = [
      '#type' => 'markup',
      '#markup' => $form_state->get('submitted_markup') ?? '',
      '#prefix' => '<div id="markup-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['#attached']['library'][] = 'ai_seo_link_advisor/styling';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Get the submitted URL and trim any whitespace.
    $url = trim($form_state->getValue('url'));

    // Parse the URL into components (scheme, host, path, etc.).
    $parsed_url = parse_url($url);

    // Check if the submitted string is a valid URL format.
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('url', $this->t('The URL is not valid.'));
      return;
    }

    // Ensure that the URL uses only 'http' or 'https' scheme.
    if (
      !empty($parsed_url['scheme']) &&
      !in_array($parsed_url['scheme'], ['http', 'https'])
    ) {
      $form_state->setErrorByName('url', $this->t('Only HTTP and HTTPS URLs are allowed.'));
      return;
    }

    // Validate the URL host against trusted_host_patterns.
    $trusted_host_patterns = $this->settings->get('trusted_host_patterns') ?? [];
    if (!empty($trusted_host_patterns)) {
      $valid = FALSE;

      foreach ($trusted_host_patterns as $pattern) {
        if (preg_match('#' . $pattern . '#', $parsed_url['host'])) {
          $valid = TRUE;
          break;
        }
      }

      if (!$valid) {
        $form_state->setErrorByName('url', $this->t('The host of the provided URL is not trusted. Please configure the trusted_host_patterns in your settings.php file to ensure proper host validation.'));
        return;
      }
    }

    // Use '/' as the default path if it's not defined in the URL.
    $path = $parsed_url['path'] ?? '/';

    // Convert the path into a Drupal-relative URL object.
    $url_object = Url::fromUserInput($path, ['absolute' => FALSE]);

    // Check if the URL path corresponds to a routable path in Drupal.
    if (!$url_object->isRouted()) {
      $form_state->setErrorByName('url', $this->t('The URL does not resolve to a Drupal page.'));
      return;
    }

    // Verify whether the current user has access to the routed page.
    if (!$url_object->access()) {
      $form_state->setErrorByName('url', $this->t('You do not have access to this page.'));
      return;
    }

    // Disallow URLs pointing to administrative routes.
    $route = $this->routeProvider->getRouteByName($url_object->getRouteName());
    if ($route->getOption('_admin_route') || str_starts_with($path, '/admin')) {
      $form_state->setErrorByName('url', $this->t('Admin pages are not allowed.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Get the URL value from the form submission.
    $url = $form_state->getValue('url');

    $render_array = $this->aiSeoLinkAdvisor->analyze($url);
    $markup = $this->renderer->render($render_array);
    $form_state->set('submitted_markup', $markup);
    // Rebuild form to show markup.
    $form_state->setRebuild();
  }

  /**
   * AJAX callback to refresh output.
   */
  public function ajaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['output'];
  }

}
