<?php

namespace Drupal\ai_seo_link_advisor\Form;

use Drupal\ai_seo_link_advisor\Controller\SeolinkcheckController;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for seo analysis.
 */
class SeourlForm extends FormBase {

  /**
   * The settings service.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected $settings;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * Constructs a new SeourlForm object.
   *
   * @param \Drupal\Core\Site\Settings $settings
   *   The settings service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Routing\RouteProviderInterface $route_provider
   *   The route provider service.
   */
  public function __construct(
    Settings $settings,
    MessengerInterface $messenger,
    RouteProviderInterface $route_provider,
  ) {
    $this->settings = $settings;
    $this->messenger = $messenger;
    $this->routeProvider = $route_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('settings'),
      $container->get('messenger'),
      $container->get('router.route_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'seourl_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // If a URL is provided in the query string, set it as the default value.
    $input_url = $this->getRequest()->query->get('url') ?? '';
    $input_url = SeolinkcheckController::sanitizeUrl($input_url);

    // Define a text field for entering the website URL.
    $form['url'] = [
      '#type' => 'url',
      '#title' => $this->t('Enter Website URL'),
      '#required' => TRUE,
      '#default_value' => $input_url,
    ];

    $trusted_host_patterns = $this->settings->get('trusted_host_patterns') ?? [];
    if (empty($trusted_host_patterns)) {
      $this->messenger->addError($this->t('Please configure the trusted_host_patterns in your settings.php file to proceed.'));
    }

    // Define the actions section with a submit button.
    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('SEO Analyze'),
      '#disabled' => empty($trusted_host_patterns),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Get the submitted URL and trim any whitespace.
    $url = trim($form_state->getValue('url'));

    // Parse the URL into components (scheme, host, path, etc.).
    $parsed_url = parse_url($url);

    // Check if the submitted string is a valid URL format.
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('url', $this->t('The URL is not valid.'));
      return;
    }

    // Ensure that the URL uses only http or https scheme.
    if (
      !empty($parsed_url['scheme']) &&
      !in_array($parsed_url['scheme'], ['http', 'https'])
    ) {
      $form_state->setErrorByName('url', $this->t('Only HTTP and HTTPS URLs are allowed.'));
      return;
    }

    // Validate the URL host against trusted_host_patterns.
    $trusted_host_patterns = $this->settings->get('trusted_host_patterns') ?? [];
    if (!empty($trusted_host_patterns)) {
      $valid = FALSE;

      foreach ($trusted_host_patterns as $pattern) {
        if (preg_match('#' . $pattern . '#', $parsed_url['host'])) {
          $valid = TRUE;
          break;
        }
      }

      if (!$valid) {
        $form_state->setErrorByName('url', $this->t('The host of the provided URL is not trusted. Please configure the trusted_host_patterns in your settings.php file to ensure proper host validation.'));
        return;
      }
    }

    // Use '/' as the default path if it's not defined in the URL.
    $path = $parsed_url['path'] ?? '/';

    // Convert the path into a Drupal-relative URL object.
    $url_object = Url::fromUserInput($path, ['absolute' => FALSE]);

    // Check if the URL path corresponds to a routable path in Drupal.
    if (!$url_object->isRouted()) {
      $form_state->setErrorByName('url', $this->t('The URL does not resolve to a Drupal page.'));
      return;
    }

    // Verify whether the current user has access to the routed page.
    if (!$url_object->access()) {
      $form_state->setErrorByName('url', $this->t('You do not have access to this page.'));
      return;
    }

    // Disallow URLs pointing to administrative routes.
    $route = $this->routeProvider->getRouteByName($url_object->getRouteName());
    if ($route->getOption('_admin_route') || str_starts_with($path, '/admin')) {
      $form_state->setErrorByName('url', $this->t('Admin pages are not allowed.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Get the URL value from the form submission.
    $url = $form_state->getValue('url');

    // Redirect to the analyzer route with the URL parameter.
    $form_state->setRedirect('ai_seo_link_advisor.analyzer', ['url' => $url]);
  }

}
