<?php

namespace Drupal\ai_seo_link_advisor\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * Service class for interacting with the OpenAI API.
 */
class OpenAIService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  private $client;

  /**
   * The API key.
   *
   * @var string
   */
  private $apiKey;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private $logger;

  /**
   * The key repository interface.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private $configFactory;

  /**
   * Constructs an OpenAIService object.
   *
   * @param \GuzzleHttp\Client $client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The Key Repository interface.
   */
  public function __construct(Client $client, ConfigFactoryInterface $configFactory, LoggerInterface $logger, KeyRepositoryInterface $key_repository) {
    $this->client = new Client(['base_uri' => 'https://api.openai.com/v1/']);
    $this->configFactory = $configFactory;
    $this->logger = $logger;
    $this->keyRepository = $key_repository;
  }

  /**
   * Retrieves the API key from the configuration settings.
   *
   * @return string
   *   The API key or an empty string if not found.
   */
  public function getApiKey() {
    // Load the module configuration.
    $config = $this->configFactory->get('ai_seo_link_advisor.settings');
    // Retrieve the API key name from the configuration.
    $key_name = $config->get('openai_api_key');
    // If the key name is set, fetch the actual key value.
    if ($key_name) {
      $key_name = $this->keyRepository->getKey($key_name);
      if ($key_name) {
        return $key_name->getKeyValue();
      }
    }

    return '';
  }

  /**
   * Send a request to OpenAI API to get SEO recommendation for given content.
   *
   * @param string $content
   *   The content to analyze for SEO recommendations.
   *
   * @return string|array
   *   The SEO recommendations or an error message.
   */
  public function getSeoRecommendations($content) {
    // Get the API key.
    $openai_api_key = $this->getApiKey();
    // Prepare the payload for the API request.
    $payload = [
      'model' => 'gpt-3.5-turbo',
      'messages' => [
        [
          'role' => 'system',
          'content' => 'You are an SEO expert. Provide SEO recommendations for the given content.',
        ],
        [
          'role' => 'user',
          'content' => $content,
        ],
      ],
    ];
    try {
      // Send the request to the OpenAI API.
      $response = $this->client->post('chat/completions', [
        'headers' => [
          'Authorization' => 'Bearer ' . $openai_api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
      ]);
      // Parse the JSON response.
      $data = json_decode($response->getBody()->getContents(), TRUE);
      return $data['choices'][0]['message']['content'];
    }
    catch (GuzzleException $e) {
      // Log the exception using the watchdog module.
      $this->logger->error('Failed to get SEO recommendations from OpenAI: @message', ['@message' => $e->getMessage()]);
      return [
        'error' => TRUE,
        'message' => 'Failed to get SEO recommendations from OpenAI.',
        'details' => $e->getMessage(),
      ];
    }
  }

}
