<?php

declare(strict_types=1);

namespace Drupal\ai_spam_protection\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AI spam protection settings for this site.
 */
final class AiSpamProtectionConfigForm extends ConfigFormBase {

  protected AiProviderPluginManager $aiProviderManager;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  protected function setAiProviderManager(AiProviderPluginManager $aiProviderManager) {
    $this->aiProviderManager = $aiProviderManager;
  }

  #[\Override]
  public static function create(ContainerInterface $container): static {
    $self = parent::create($container);
    $self->setAiProviderManager($container->get('ai.provider'));

    return $self;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_spam_protection_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ai_spam_protection.settings'];
  }

  public function buildForm($form, $form_state) {

    $multiline_form_config = fn ($value) => $value ? implode('\r\n', $value) : '';
    $multiline_to_config = fn ($value) => $value ? array_map('trim', explode("\n", $value)) : [];

    $form['protect_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Protect all forms'),
      '#config_target' => 'ai_spam_protection.settings:protect_all',
      '#description' => $this->t('When enabled, all the form are protected. <b>Be careful</b> with this option, all forms, included all forms related to the site configuration, will be protected by AI spam protection. This option is not recommended unless you are well aware of the implications. With this option enabled, it is highly recommended that you set the IP addresses of the administrators in the white list, or to give the Bypass permission to theses users.'),
    ];

    $form['protected_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Protected Form IDs'),
      '#config_target' => new ConfigTarget('ai_spam_protection.settings', 'protected_ids', $multiline_form_config, $multiline_to_config),
      '#description' => $this->t('Specify the form IDs that should be protected with AI spam protection. Each form ID should be on a separate line. Wildcard (*) characters can be used. Base form IDs are supported. Examples of form id : comment_*, contact_*, simplenews_subscriber_form, node_form, node_article_edit_form, etc.'),
      '#states' => [
        'invisible' => [
          ':input[name="protect_all"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['unprotected_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Unprotected form IDs'),
      '#config_target' => new ConfigTarget('ai_spam_protection.settings', 'unprotected_ids', $multiline_form_config, $multiline_to_config),
      '#description' => $this->t('Specify the form IDs that should be <b>NOT</b> protected With AI spam protection. Each form ID should be on a separate line. Wildcard (*) characters can be used. Base form IDs are supported.'),
      '#states' => [
        'visible' => [
          ':input[name="protect_all"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['whitelist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed IP addresses'),
      '#config_target' => new ConfigTarget('ai_spam_protection.settings', 'whitelist', $multiline_form_config, $multiline_to_config),
      '#description' => $this->t('Specify IP addresses that should bypass AI spam protection. Each IP address should be on a separate line. Wildcard (*) characters can be used (for example, 192.0.2.*).'),
    ];

    $form['flood_control'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Flood control'),
      '#config_target' => 'ai_spam_protection.settings:flood_control',
      '#description' => $this->t('Log submissions marked as spam as flood attempt. When too many attempts are done, submissions are blocked without sending them to the AI. This protects your AI from being flooded with spam protection checks.'),
    ];

    $form['flood_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Window'),
      '#min' => 1,
      '#config_target' => 'ai_spam_protection.settings:flood_window',
      '#description' => $this->t('The window of the flood control, in seconds.'),
      '#states' => [
        'visible' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['flood_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Threshold'),
      '#min' => 1,
      '#config_target' => 'ai_spam_protection.settings:flood_threshold',
      '#description' => $this->t('The threshold of the flood control.'),
      '#states' => [
        'visible' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Error message'),
      '#description' => $this->t('The error message to display when a form submission has been identified as spam.'),
      '#required' => TRUE,
      '#config_target' => 'ai_spam_protection.settings:error_message',
    ];

    $form['human_interaction'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow human interaction'),
      '#description' => $this->t('When the AI flags a form submission as spam, allow a human to check a checkbox to confirm it is not spam. This will then bypass the AI check.'),
      '#config_target' => 'ai_spam_protection.settings:human_interaction',
    ];

    $form['checkbox_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Checkbox label'),
      '#description' => $this->t('When the spam check fails, we give the option to override the check by showing the end user a checkbox to confirm it is not spam. This way the end user can filter out false positives from the AI, but spam bots are still stopped.'),
      '#config_target' => 'ai_spam_protection.settings:checkbox_label',
      '#states' => [
        'visible' => [
          ':input[name="human_interaction"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="human_interaction"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['provider_model'] = [
      '#type' => 'select',
      '#title' => $this->t('Custom Model'),
      '#description' => $this->t('Choose a provider/model if you are not using the default "chat" model'),
      '#options' => $this->aiProviderManager->getSimpleProviderModelOptions('chat'),
      '#config_target' => 'ai_spam_protection.settings:provider_model'
    ];

    $form['prompt'] = [
      '#title' => $this->t('Prompt'),
      '#type' => 'ai_prompt',
      '#prompt_types' => ['ai_spam_protection'],
      '#required' => TRUE,
      '#config_target' => 'ai_spam_protection.settings:prompt'
    ];

    return parent::buildForm($form, $form_state);
  }
}
