<?php

namespace Drupal\ai_tool_get_ip_address\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Plugin implementation to get user's IP address tool.
 */
#[Tool(
  id: 'ai_tool_get_ip_address',
  label: new TranslatableMarkup('Get IP Address'),
  description: new TranslatableMarkup('Returns the IP address of the user.'),
  operation: ToolOperation::Read,
)]
class GetIPAddress extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->requestStack = $container->get('request_stack');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    $request = $this->requestStack->getCurrentRequest();
    if (!$request) {
      return ExecutableResult::failure($this->t('No active request available.'));
    }

    // getClientIp respects trusted proxies configuration.
    $ip = $request->getClientIp();
    $ips = $request->getClientIps();

    return ExecutableResult::success($this->t('Success retrieving IP address.'), [
      'result' => [
        'IP' => $ip,
        'All IPS' => $ips,
      ],
    ]);
  }

  /**
   * Access check.
   *
   * @param array $values
   *   The data values.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   User account.
   * @param bool $return_as_object
   *   If the return should be an object.
   *
   * @return bool|\Drupal\Core\Access\AccessResultInterface
   *   The access return.
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
