<?php

namespace Drupal\ai_tool_get_sitemap\Plugin\AiFunctionCall;

use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai_agents\PluginInterfaces\AiAgentContextInterface;
use Drupal\simple_sitemap\Manager\Generator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation to get the sitemap.
 */
#[FunctionCall(
  id: 'ai_agents::ai_agent:get_sitemap',
  function_name: 'ai_agent_get_sitemap',
  name: 'Get Sitemap',
  description: 'Returns the sitemap of the website.',
  group: 'information_tools',
  context_definitions: [
  ],
)]
class GetSitemap extends FunctionCallBase implements ExecutableFunctionCallInterface, AiAgentContextInterface {

  /**
   * The config data.
   *
   * @var string
   */
  protected string $response = '';

  /**
   * The Simple XML sitemap generator.
   *
   * @var \Drupal\simple_sitemap\Manager\Generator $simpleSitemapGenerator
   */
  protected ?Generator $simpleSitemapGenerator;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai.context_definition_normalizer'),
    );
    // Check if the service exists before trying to get it.
    if ($container->has('simple_sitemap.generator')) {
      $instance->simpleSitemapGenerator = $container->get('simple_sitemap.generator');
    }
    else {
      $instance->simpleSitemapGenerator = NULL;
    }
    return $instance;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\ai_agents\Exception\AgentProcessingException
   */
  public function execute() {
    $output = 'No sitemap available. Please install the Simple XML Sitemap module.';
    if ($this->simpleSitemapGenerator) {
      $defaultSitemap = $this->simpleSitemapGenerator->getDefaultSitemap();
      $variant = $defaultSitemap?->id();
      $output = $this->simpleSitemapGenerator->setSitemaps($variant)
        ->getContent();
    }
    $this->response = json_encode($output);
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableOutput(): string {
    return $this->response;
  }

}
