<?php

namespace Drupal\ai_tool_get_sitemap\Plugin\tool\Tool;

use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\simple_sitemap\Manager\Generator;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation to get the sitemap.
 */
#[Tool(
  id: 'ai_tool_get_sitemap',
  label: new TranslatableMarkup('Get Sitemap'),
  description: new TranslatableMarkup('Returns the sitemap of the website.'),
  input_definitions: [],
  output_definitions: [
    'sitemap' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('Sitemap Info'),
      description: new TranslatableMarkup('The Sitemap.')
    ),
  ],
)]
class GetSitemap extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * The constructor.
   *
   * @param array $configuration
   * @param $plugin_id
   * @param $plugin_definition
   * @param \Drupal\simple_sitemap\Manager\Generator $simpleSitemapGenerator
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected ?Generator $simpleSitemapGenerator
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->has('simple_sitemap.generator') ? $container->get('simple_sitemap.generator') : NULL
    );

    return $instance;
  }

  protected function doExecute(array $values): ExecutableResult {
    $output = 'No sitemap available. Please install the Simple XML Sitemap module.';
    if ($this->simpleSitemapGenerator) {
      $defaultSitemap = $this->simpleSitemapGenerator->getDefaultSitemap();
      $variant = $defaultSitemap?->id();
      $output = $this->simpleSitemapGenerator->setSitemaps($variant)
        ->getContent();
      return ExecutableResult::success($this->t('Success retrieving IP address.'), [
        'sitemap' => $output,
      ]);
    }

    return ExecutableResult::failure($this->t($output), []);
  }

  /**
   * @param array $values
   * @param \Drupal\Core\Session\AccountInterface|null $account
   * @param $return_as_object
   *
   * @return bool|\Drupal\Core\Access\AccessResultInterface
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|\Drupal\Core\Access\AccessResultInterface {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
