<?php

namespace Drupal\ai_tool_get_sitemap\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\simple_sitemap\Manager\Generator;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation to get the sitemap.
 */
#[Tool(
  id: 'ai_tool_get_sitemap',
  label: new TranslatableMarkup('Get Sitemap'),
  description: new TranslatableMarkup('Returns the sitemap of the website.'),
)]
class GetSitemap extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The simple sitemap generator.
   *
   * @var \Drupal\simple_sitemap\Manager\Generator
   */
  protected Generator $simpleSitemapGenerator;

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->simpleSitemapGenerator = $container->has('simple_sitemap.generator') ? $container->get('simple_sitemap.generator') : NULL;
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    $output = 'No sitemap available. Please install the Simple XML Sitemap module.';
    if ($this->simpleSitemapGenerator) {
      $defaultSitemap = $this->simpleSitemapGenerator->getDefaultSitemap();
      $variant = $defaultSitemap?->id();
      $output = $this->simpleSitemapGenerator->setSitemaps($variant)
        ->getContent();
      return ExecutableResult::success($this->t('Success retrieving IP address.'), [
        'result' => $output,
      ]);
    }

    return ExecutableResult::failure($this->t('Cannot get the sitemap. @error', ['@error' => $output]));
  }

  /**
   * Access check.
   *
   * @param array $values
   *   The data values.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   User account.
   * @param bool $return_as_object
   *   If the return should be an object.
   *
   * @return bool|\Drupal\Core\Access\AccessResultInterface
   *   The access return.
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
