<?php

namespace Drupal\ai_tool_get_sitemap\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\simple_sitemap\Manager\Generator;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation to get the sitemap.
 */
#[Tool(
  id: 'ai_tool_get_sitemap',
  label: new TranslatableMarkup('Get Sitemap'),
  description: new TranslatableMarkup('Returns the sitemap of the website.'),
  operation: ToolOperation::Read,
  output_definitions: [
    'sitemap' => new ContextDefinition(
      data_type: 'map',
      label: new TranslatableMarkup("Sitemap."),
      description: "The sitemap.",
    ),
  ]
)]
class GetSitemap extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * The simple sitemap generator.
   *
   * @var \Drupal\simple_sitemap\Manager\Generator
   */
  protected ?Generator $simpleSitemapGenerator = NULL;

  /**
   * Factory method to create the plugin instance from the service container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array<mixed> $configuration
   *   The plugin configuration array.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition (usually an array).
   *
   * @return \Drupal\ai\Service\FunctionCalling\FunctionCallInterface|static
   *   The created plugin instance.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    // Avoid a ternary here to silence PHPStan's "ternary always true" warning.
    if ($container->has('simple_sitemap.generator')) {
      $instance->simpleSitemapGenerator = $container->get('simple_sitemap.generator');
    }
    else {
      $instance->simpleSitemapGenerator = NULL;
    }
    return $instance;
  }

  /**
   * Execute.
   *
   * @param array<mixed> $values
   *   The parameters.
   *
   * @return \Drupal\tool\ExecutableResult
   *   Returns results.
   */
  protected function doExecute(array $values): ExecutableResult {
    $output = 'No sitemap available. Please install the Simple XML Sitemap module.';
    $defaultSitemap = $this->simpleSitemapGenerator->getDefaultSitemap();
    $variant = $defaultSitemap?->id();

    // Set generator to the desired sitemap(s) and get the XML content.
    // Ensure the argument matches the expected types: null|string|array<...>.
    $sitemapsArg = $variant === NULL ? NULL : (string) $variant;

    $sitemap = $this->simpleSitemapGenerator->setSitemaps($sitemapsArg);
    $xmlContent = $sitemap->getContent();
    $xmlContentCount = $sitemap->getDefaultSitemap()->getLinkCount();

    if ($xmlContentCount) {
      return ExecutableResult::success($this->t('Success retrieving the sitemap.'), [
        'sitemap' => [$xmlContent],
        'result' => [$xmlContent],
      ]);
    }
    return ExecutableResult::failure($this->t('Cannot get the sitemap. @error', ['@error' => $output]));
  }

  /**
   * Access check.
   *
   * @param array<mixed> $values
   *   The data values.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   User account.
   * @param bool $return_as_object
   *   If the return should be an object.
   *
   * @return bool|\Drupal\Core\Access\AccessResultInterface
   *   The access return.
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
