<?php

namespace Drupal\ai_translate_plus\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\ai_translate_plus\Service\TokenReplacementService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for configuring bundle-specific prompts.
 */
class BundlePromptForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The token replacement service.
   *
   * @var \Drupal\ai_translate_plus\Service\TokenReplacementService
   */
  protected TokenReplacementService $tokenReplacementService;

  /**
   * Constructs a new BundlePromptForm object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    TokenReplacementService $token_replacement_service
  ) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->tokenReplacementService = $token_replacement_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('ai_translate_plus.token_replacement')
    );
  }

  /**
   * Gets a list of entity types that support translation.
   *
   * @return array
   *   An array of translatable entity types.
   */
  protected function getTranslatableEntityTypes(): array {
    $entity_types = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->entityClassImplements('\Drupal\Core\Entity\ContentEntityInterface')) {
        if ($entity_type->isTranslatable()) {
          $entity_types[$entity_type_id] = $entity_type->getLabel();
        }
      }
    }
    return $entity_types;
  }

  /**
   * Gets a list of bundles for an entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   An array of bundles.
   */
  protected function getBundlesForEntityType(string $entity_type_id): array {
    $bundles = [];
    try {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $bundle_entity_type = $entity_type->getBundleEntityType();

      if ($bundle_entity_type) {
        $bundle_storage = $this->entityTypeManager->getStorage($bundle_entity_type);
        foreach ($bundle_storage->loadMultiple() as $bundle_entity) {
          $bundles[$bundle_entity->id()] = $bundle_entity->label();
        }
      }
      else {
        // Some entity types don't have bundles, so use the entity type ID as the bundle
        $bundles[$entity_type_id] = $entity_type->getLabel();
      }
    }
    catch (\Exception $e) {
      // Log and continue
      \Drupal::logger('ai_translate_plus')->error('Error getting bundles: @message', ['@message' => $e->getMessage()]);
    }
    return $bundles;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_translate_plus_bundle_prompt_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_translate_plus.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_translate_plus.settings');
    $bundle_prompts = $config->get('prompts') ?? [];

    // Entity type selection
    $entity_types = $this->getTranslatableEntityTypes();

    $selected_entity_type = $form_state->getValue('entity_type');
    if (!$selected_entity_type && !empty($entity_types)) {
      // Default to first entity type
      $selected_entity_type = key($entity_types);
      $form_state->setValue('entity_type', $selected_entity_type);
    }

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity type'),
      '#options' => $entity_types,
      '#required' => TRUE,
      '#default_value' => $selected_entity_type,
    ];

    $form['bundle_form_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'bundle-form-wrapper'],
    ];

    if ($selected_entity_type) {
      // Get bundles for this entity type
      $bundles = $this->getBundlesForEntityType($selected_entity_type);

      // Get languages
      $languages = $this->languageManager->getLanguages();
      $default_langcode = $this->languageManager->getDefaultLanguage()->getId();

      $form['bundle_form_wrapper']['bundle'] = [
        '#type' => 'select',
        '#title' => $this->t('Bundle'),
        '#options' => $bundles,
        '#required' => TRUE,
      ];

      $selected_bundle = $form_state->getValue('bundle');
      if (!$selected_bundle && !empty($bundles)) {
        // Default to first bundle
        $selected_bundle = key($bundles);
        $form_state->setValue('bundle', $selected_bundle);
        $form['bundle_form_wrapper']['bundle']['#default_value'] = $selected_bundle;
      }

      $form['bundle_form_wrapper']['prompt_form_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'prompt-form-wrapper'],
      ];

      if ($selected_bundle) {
        $form['bundle_form_wrapper']['prompt_form_wrapper']['prompts'] = [
          '#type' => 'details',
          '#title' => $this->t('Prompts for @entity_type - @bundle', [
            '@entity_type' => $entity_types[$selected_entity_type],
            '@bundle' => $bundles[$selected_bundle],
          ]),
          '#open' => TRUE,
          '#tree' => TRUE,
        ];

        foreach ($languages as $language) {
          $langcode = $language->getId();

          // Skip default language
          if ($langcode === $default_langcode) {
            continue;
          }

          $prompt = $bundle_prompts[$selected_entity_type][$selected_bundle][$langcode] ?? '';

          $description = $this->t('The prompt to use when translating @bundle entities to @language. Leave empty to use the entity-type default prompt or the global prompt.', [
            '@bundle' => $bundles[$selected_bundle],
            '@language' => $language->getName(),
          ]);

          // Add simple token help
          $token_help = $this->tokenReplacementService->getTokenHelpText($selected_entity_type);
          if ($token_help) {
            $description .= '<br><br>' . $token_help;
          }

          $form['bundle_form_wrapper']['prompt_form_wrapper']['prompts'][$langcode] = [
            '#type' => 'textarea',
            '#title' => $this->t('Prompt for @language translation', ['@language' => $language->getName()]),
            '#default_value' => $prompt,
            '#rows' => 5,
            '#description' => $description,
          ];
        }
      }
    }

    // Add submit button
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_translate_plus.settings');
    $bundle_prompts = $config->get('prompts') ?? [];

    $entity_type_id = $form_state->getValue('entity_type');
    $bundle = $form_state->getValue('bundle');
    $prompts = $form_state->getValue('prompts');

    if ($entity_type_id && $bundle && $prompts) {
      foreach ($prompts as $langcode => $prompt) {
        if (!empty($prompt)) {
          $bundle_prompts[$entity_type_id][$bundle][$langcode] = $prompt;
        }
        elseif (isset($bundle_prompts[$entity_type_id][$bundle][$langcode])) {
          unset($bundle_prompts[$entity_type_id][$bundle][$langcode]);

          // Clean up empty arrays
          if (empty($bundle_prompts[$entity_type_id][$bundle])) {
            unset($bundle_prompts[$entity_type_id][$bundle]);

            if (empty($bundle_prompts[$entity_type_id])) {
              unset($bundle_prompts[$entity_type_id]);
            }
          }
        }
      }

      $config->set('prompts', $bundle_prompts)->save();
      $this->messenger()->addStatus($this->t('The prompts have been saved.'));
    }
  }

}
