<?php

namespace Drupal\ai_translate_plus\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\ai_translate_plus\Service\TokenReplacementService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for configuring entity-type level prompts.
 */
class EntityTypePromptForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The token replacement service.
   *
   * @var \Drupal\ai_translate_plus\Service\TokenReplacementService
   */
  protected TokenReplacementService $tokenReplacementService;

  /**
   * Constructs a new EntityTypePromptForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\ai_translate_plus\Service\TokenReplacementService $token_replacement_service
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    TokenReplacementService $token_replacement_service
  ) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->tokenReplacementService = $token_replacement_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('ai_translate_plus.token_replacement')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_translate_plus_entity_type_prompt_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_translate_plus.settings'];
  }

  /**
   * Gets a list of entity types that support translation.
   *
   * @return array
   *   An array of translatable entity types.
   */
  protected function getTranslatableEntityTypes(): array {
    $entity_types = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->entityClassImplements('\Drupal\Core\Entity\ContentEntityInterface')) {
        if ($entity_type->isTranslatable()) {
          $entity_types[$entity_type_id] = $entity_type->getLabel();
        }
      }
    }
    return $entity_types;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_translate_plus.settings');
    $entity_type_prompts = $config->get('entity_type_prompts') ?? [];

    // Get translatable entity types
    $entity_types = $this->getTranslatableEntityTypes();

    // Get languages
    $languages = $this->languageManager->getLanguages();
    $default_langcode = $this->languageManager->getDefaultLanguage()->getId();

    $form['description'] = [
      '#markup' => $this->t('<p>Configure default prompts for each entity type. These prompts will be used when no bundle-specific prompt is defined.</p>'),
    ];

    $form['entity_type_prompts'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Entity type prompts'),
    ];

    foreach ($entity_types as $entity_type_id => $entity_type_label) {
      $form[$entity_type_id] = [
        '#type' => 'details',
        '#title' => $entity_type_label,
        '#group' => 'entity_type_prompts',
        '#tree' => TRUE,
      ];

      foreach ($languages as $language) {
        $langcode = $language->getId();

        // Skip default language
        if ($langcode === $default_langcode) {
          continue;
        }

        $prompt = $entity_type_prompts[$entity_type_id][$langcode] ?? '';

        $description = $this->t('The prompt to use when translating @entity_type entities to @language. Leave empty to use the global prompt.', [
          '@entity_type' => $entity_type_label,
          '@language' => $language->getName(),
        ]);

        // Add simple token help
        $token_help = $this->tokenReplacementService->getTokenHelpText($entity_type_id);
        if ($token_help) {
          $description .= '<br><br>' . $token_help;
        }

        $form[$entity_type_id][$langcode] = [
          '#type' => 'textarea',
          '#title' => $this->t('Prompt for @language translation', ['@language' => $language->getName()]),
          '#default_value' => $prompt,
          '#rows' => 5,
          '#description' => $description,
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_translate_plus.settings');
    $entity_type_prompts = $config->get('entity_type_prompts') ?? [];

    // Get translatable entity types
    $entity_types = $this->getTranslatableEntityTypes();

    // Get languages
    $languages = $this->languageManager->getLanguages();
    $default_langcode = $this->languageManager->getDefaultLanguage()->getId();

    foreach ($entity_types as $entity_type_id => $entity_type_label) {
      foreach ($languages as $language) {
        $langcode = $language->getId();

        // Skip default language
        if ($langcode === $default_langcode) {
          continue;
        }

        $prompt = $form_state->getValue([$entity_type_id, $langcode]);

        if (!empty($prompt)) {
          $entity_type_prompts[$entity_type_id][$langcode] = $prompt;
        }
        elseif (isset($entity_type_prompts[$entity_type_id][$langcode])) {
          unset($entity_type_prompts[$entity_type_id][$langcode]);

          // Clean up empty arrays
          if (empty($entity_type_prompts[$entity_type_id])) {
            unset($entity_type_prompts[$entity_type_id]);
          }
        }
      }
    }

    $config->set('entity_type_prompts', $entity_type_prompts)->save();

    parent::submitForm($form, $form_state);
  }

}
