<?php

declare(strict_types=1);

namespace Drupal\ai_vdb_provider_opensearch\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\search_api_opensearch\Connector\ConnectorFormTrait;
use Drupal\search_api_opensearch\Connector\ConnectorPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring OpenSearch settings.
 */
class OpenSearchConfigForm extends FormBase {

  use ConnectorFormTrait;
  use MessengerTrait;

  /**
   * The default configuration.
   */
  protected array $configuration = [
    'connector' => 'standard',
    'connector_config' => [],
  ];

  public function __construct(
    ConfigFactoryInterface $configFactory,
    MessengerInterface $messenger,
    protected ConnectorPluginManager $connectorPluginManager,
  ) {
    $this->setConfigFactory($configFactory);
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('plugin.manager.search_api_opensearch.connector'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_vdb_provider_opensearch_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('ai_vdb_provider_opensearch.settings');
    $data = $config->getRawData() === [] ? $this->configuration : $config->getRawData();
    $this->buildConnectorConfigForm($form, $form_state, $data);
    $form['vdb_config'] = [
      '#type' => 'details',
      '#title' => $this->t('Configure vector database'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];
    $form['vdb_config']['engine'] = [
      '#title' => $this->t('Engine'),
      '#description' => 'See <a target="_blank" href="https://docs.opensearch.org/latest/field-types/supported-field-types/knn-methods-engines/">OpenSearch Documentation</a> for details.',
      '#type' => 'select',
      '#required' => TRUE,
      '#default_value' => $data['vdb_config']['engine'] ?? 'faiss',
      '#options' => [
        'faiss' => 'FAISS (Default)',
        'lucene' => 'Lucene',
        'nmslib' => 'NMSLIB (Deprecated, incompatible with OpenSearch 3 and above)',
      ],
    ];
    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
      '#button_type' => 'primary',
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('ai_vdb_provider_opensearch.settings');
    $this->validateConnectorConfigForm($form, $form_state, $config->getRawData());
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->submitConnectorConfigForm($form, $form_state);
    $config = $this->configFactory->getEditable('ai_vdb_provider_opensearch.settings');
    $config->set('connector', $form_state->getValue('connector'));
    $config->set('connector_config', $form_state->getValue('connector_config'));
    $config->set('vdb_config', $form_state->getValue('vdb_config'));
    $config->save();
    $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
  }

  /**
   * Handles switching the selected connector plugin.
   */
  public static function buildAjaxConnectorConfigForm(array $form, FormStateInterface $form_state): array {
    // The work is already done in form(), where we rebuild the entity according
    // to the current form values and then create the backend configuration form
    // based on that. So we just need to return the relevant part of the form
    // here.
    return $form['connector_config'];
  }

}
