<?php

declare(strict_types=1);

// cspell:ignore hnsw nmslib cosinesimil innerproduct
namespace Drupal\ai_vdb_provider_opensearch\Plugin\VdbProvider;

use Drupal\ai\Attribute\AiVdbProvider;
use Drupal\ai\Base\AiVdbProviderClientBase;
use Drupal\ai\Enum\VdbSimilarityMetrics;
use Drupal\Component\Plugin\DependentPluginInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\key\KeyRepositoryInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_opensearch\Connector\ConnectorFormTrait;
use Drupal\search_api_opensearch\Connector\ConnectorPluginManager;
use Drupal\search_api_opensearch\SearchAPI\Query\FilterBuilder;
use OpenSearch\Client;
use OpenSearch\Exception\BadRequestHttpException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Plugin implementation of the OpenSearch VDB provider.
 */
#[AiVdbProvider(
  id: 'opensearch',
  label: new TranslatableMarkup('OpenSearch Vector DB'),
)]
class OpenSearchVdbProvider extends AiVdbProviderClientBase implements DependentPluginInterface {

  use StringTranslationTrait;
  use ConnectorFormTrait;

  /**
   * Configuration for the OpenSearch client.
   *
   * @var array<string, mixed>
   */
  protected array $clientConfig = [];

  /**
   * The OpenSearch client.
   */
  protected Client $client;

  public function __construct(
    string $plugin_id,
    mixed $plugin_definition,
    ConfigFactoryInterface $configFactory,
    KeyRepositoryInterface $keyRepository,
    EventDispatcherInterface $eventDispatcher,
    EntityFieldManagerInterface $entityFieldManager,
    MessengerInterface $messenger,
    protected ConnectorPluginManager $connectorPluginManager,
    protected FilterBuilder $filterBuilder,
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $configFactory,
      $keyRepository,
      $eventDispatcher,
      $entityFieldManager,
      $messenger,
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): AiVdbProviderClientBase | static {
    return new static(
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('event_dispatcher'),
      $container->get('entity_field.manager'),
      $container->get('messenger'),
      $container->get('plugin.manager.search_api_opensearch.connector'),
      $container->get(FilterBuilder::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getClient(): Client {
    if (!isset($this->client)) {
      $config = $this->getConfig()->get();
      $this->client = $this->getConnector($config['connector'], $config['connector_config'])->getClient();
    }
    return $this->client;
  }

  /**
   * {@inheritdoc}
   */
  public function buildSettingsForm(array $form, FormStateInterface $form_state, array $configuration): array {
    $form = parent::buildSettingsForm($form, $form_state, $configuration);
    $this->buildConnectorConfigForm($form, $form_state, $this->getConfig()->getOriginal('', FALSE));
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateSettingsForm(array &$form, FormStateInterface $form_state): void {
    // We need to store configuration here so we can access it when
    // building the client.
    $this->clientConfig = $form_state->getValue('database_settings', []);
    parent::validateSettingsForm($form, $form_state);
    $form_state->setValue('connector_config', $this->clientConfig['connector_config'] ?? []);
    $form_state->setValue('connector', $this->clientConfig['connector'] ?? 'standard');
    // Temporarily add the connector and config to the form so it can be
    // validated.
    $form['connector'] = $form['database_settings']['connector'] ?? [];
    $form['connector_config'] = $form['database_settings']['connector_config'] ?? [];
    $this->validateConnectorConfigForm($form, $form_state, $this->getConfig()->getOriginal('', FALSE));
    // Remove it again.
    unset($form['connector'], $form['connector_config']);
  }

  /**
   * {@inheritdoc}
   */
  public function submitSettingsForm(array &$form, FormStateInterface $form_state): void {
    parent::submitSettingsForm($form, $form_state);
    $this->submitConnectorConfigForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_vdb_provider_opensearch.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function ping(): bool {
    try {
      $this->getClient()->ping();
      return TRUE;
    }
    catch (\Exception) {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isSetup(): bool {
    try {
      $this->getClient();
      return TRUE;
    }
    catch (\Exception) {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCollections(string $database = 'default'): array {
    $response = $this->getClient()->list()->indices();
    return array_map(fn($index) => $index['index'], $response['indices']);
  }

  /**
   * {@inheritdoc}
   */
  public function createCollection(string $collection_name, int $dimension, VdbSimilarityMetrics $metric_type = VdbSimilarityMetrics::EuclideanDistance, string $database = 'default'): void {
    $index_name = $this->formatIndexName($database, $collection_name);
    $config = $this->getConfig()->get();
    $engine = $config['vdb_config']['engine'];
    try {
      $this->getClient()->indices()->create([
        'index' => $index_name,
        'body' => [
          'settings' => [
            'index' => [
              'knn' => TRUE,
            ],
          ],
          'mappings' => [
            'properties' => [
              'vector' => [
                'type' => 'knn_vector',
                'dimension' => $dimension,
                'method' => [
                  'name' => 'hnsw',
                  'space_type' => $this->mapMetricType($metric_type),
                  'engine' => $engine,
                  'parameters' => [
                    'ef_construction' => 128,
                    'm' => 16,
                  ],
                ],
              ],
            ],
          ],
        ],
      ]);
    }
    catch (BadRequestHttpException $e) {
      // Consider making this a warning or informational
      // as this will likely only happen if the index already exists, which is
      // fine.
      $this->getLogger(
        'ai_vdb_provider_opensearch'
      )->error(
        'Failed to create OpenSearch index %index: %message',
        [
          '%index' => $index_name,
          '%message' => $e->getMessage(),
        ]
      );
    }

  }

  /**
   * {@inheritdoc}
   */
  public function dropCollection(string $collection_name, string $database = 'default'): void {
    $index_name = $this->formatIndexName($database, $collection_name);
    $this->getClient()->indices()->delete([
      'index' => $index_name,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function insertIntoCollection(string $collection_name, array $data, string $database = 'default'): void {
    $index_name = $this->formatIndexName($database, $collection_name);

    $id = $data['id'] ?? uniqid();
    $doc_data = $data;
    if (isset($doc_data['id'])) {
      unset($doc_data['id']);
    }
    // Ensure vector data is properly formatted.
    if (isset($data['vector']) && is_array($data['vector'])) {
      $doc_data['vector'] = array_map('floatval', $data['vector']);
    }
    $this->getClient()->index([
      'index' => $index_name,
      'id' => $id,
      'body' => $doc_data,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteFromCollection(string $collection_name, array $ids, string $database = 'default'): void {
    $index_name = $this->formatIndexName($database, $collection_name);
    $this->getClient()->bulk(
      [
        'index' => $index_name,
        'body' => array_map(function ($id) {
          return [
            'delete' => [
              '_id' => $id,
            ],
          ];
        }, $ids),
      ]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function querySearch(string $collection_name, array $output_fields, string $filters = '', int $limit = 10, int $offset = 0, string $database = 'default'): array {
    throw new \BadMethodCallException('Not implemented');
  }

  /**
   * {@inheritdoc}
   */
  public function vectorSearch(string $collection_name, array $vector_input, array $output_fields, QueryInterface $query, string $filters = '', int $limit = 10, int $offset = 0, string $database = 'default'): array {
    $index = $this->formatIndexName($database, $collection_name);

    // Build the knn query for OpenSearch.
    $result = $this->getClient()->search([
      'index' => $index,
      'body' => [
        'size' => $limit,
        'from' => $offset,
        'query' => [
          'knn' => [
            'vector' => [
              'vector' => $vector_input,
              'k' => $limit,
            ],
          ],
        ],
        '_source' => $output_fields,
      ],
    ]);
    return $this->formatSearchResults($result)['data'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getVdbIds(string $collection_name, array $drupalIds, string $database = 'default'): array {
    throw new \BadMethodCallException('Not implemented');
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(array $configuration, array $item_ids): void {
    $database = $configuration['database_settings']['database_name'] ?? 'default';
    $collection = $configuration['database_settings']['collection'] ?? 'default';
    $index = $this->formatIndexName($database, $collection);
    $this->getClient()->deleteByQuery(
      [
        'index' => $index,
        'body' => [
          'query' => [
            'terms' => [
              'drupal_entity_id' => array_values($item_ids),
            ],
          ],
        ],
      ]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function prepareFilters(QueryInterface $query): mixed {
    $conditionGroup = $query->getConditionGroup();
    return $this->filterBuilder->buildFilters($conditionGroup, $query->getIndex()
      ->getFields());
  }

  /**
   * Maps metric types to OpenSearch space types.
   */
  protected function mapMetricType(VdbSimilarityMetrics $metricType): string {
    return match ($metricType) {
      VdbSimilarityMetrics::EuclideanDistance => 'l2',
      VdbSimilarityMetrics::CosineSimilarity => 'cosinesimil',
      VdbSimilarityMetrics::InnerProduct => 'innerproduct',
    };
  }

  /**
   * Format an index name according to OpenSearch requirements.
   *
   * @param string $database_name
   *   The database name prefix.
   * @param string $collection_name
   *   The collection name.
   *
   * @return string
   *   Properly formatted OpenSearch index name.
   */
  protected function formatIndexName(string $database_name, string $collection_name): string {
    $index_name = strtolower($database_name . '_' . $collection_name);
    $index_name = preg_replace('/[^a-z0-9_]/', '_', $index_name);

    // Ensure the index name doesn't start with an underscore (OpenSearch
    // restriction)
    if (str_starts_with($index_name, '_')) {
      $index_name = 'idx' . $index_name;
    }

    return $index_name;
  }

  /**
   * {@inheritdoc}
   */
  public static function buildAjaxConnectorConfigForm(array $form, FormStateInterface $form_state): array {
    return [];
  }

  /**
   * Format OpenSearch search results to match the expected structure.
   *
   * @param array $results
   *   The raw OpenSearch search results.
   *
   * @return array
   *   The formatted search results.
   */
  protected function formatSearchResults(array $results): array {
    $formatted = [
      // Success code.
      'code' => 0,
      'data' => [],
    ];

    if (!empty($results['hits']['hits'])) {
      foreach ($results['hits']['hits'] as $hit) {
        $doc = $hit['_source'];
        // Add _id to the document as 'id' if not already present.
        $doc['id'] = $hit['_id'];
        // Add score if available.
        if (isset($hit['_score'])) {
          $doc['score'] = $hit['_score'];
          // SearchApiBackend expects a 'distance' field for vector search
          // results relevancy score.
          $doc['distance'] = $hit['_score'];
        }
        $formatted['data'][] = $doc;
      }
    }

    return $formatted;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    return [
      'config' => [
        'ai_vdb_provider_opensearch.settings',
      ],
    ];
  }

}
