<?php

declare(strict_types=1);

namespace Drupal\Tests\ai_vdb_provider_opensearch;

use Drupal\Tests\BrowserTestBase;
use PHPUnit\Framework\Attributes\CoversNothing;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for the OpenSearch VDB provider.
 */
#[Group('ai_vdb_provider_opensearch')]
#[CoversNothing]
class OpenSearchVdbProviderTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ai',
    'search_api_opensearch',
    'ai_vdb_provider_opensearch',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create an admin user.
    $admin_user = $this->drupalCreateUser([
      'access administration pages',
      'administer ai providers',
    ]);
    $this->drupalLogin($admin_user);
  }

  /**
   * Tests the OpenSearch VDB provider configuration form.
   */
  public function testOpenSearchVdbProviderConfiguration(): void {
    // Navigate to the AI VDB providers configuration page.
    $this->drupalGet('admin/config/ai/vdb_providers');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->linkExists('OpenSearch Configuration');

    // Click on "Configure OpenSearch".
    $this->clickLink('OpenSearch Configuration');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Configure OpenSearch');

    // Select "Standard" from the connector radio buttons.
    $this->getSession()->getPage()->selectFieldOption('connector', 'standard');
    $this->assertSession()->pageTextContains('Standard');

    // Fill in the configuration form.
    $edit = [
      'connector' => 'standard',
      'connector_config[url]' => 'http://localhost:9200',
      'vdb_config[engine]' => 'faiss',
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify that the settings were saved successfully.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    $config = \Drupal::config('ai_vdb_provider_opensearch.settings');
    $this->assertEquals('standard', $config->get('connector'));
    $this->assertEquals('http://localhost:9200', $config->get('connector_config.url'));
    $this->assertEquals('faiss', $config->get('vdb_config.engine'));
  }

}
