<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for the tool used to create web forms.
 */
#[Tool(
  id: 'ai_webform_agent:create_webform',
  label: new TranslatableMarkup('Create Webform'),
  description: new TranslatableMarkup('This creates a new webform with the specified title, optional description and url.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'webform_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform ID"),
      description: new TranslatableMarkup("The machine name of the webform to create. If not provided, a machine name will be generated from the title."),
      required: FALSE,
    ),
    'title' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform Title"),
      description: new TranslatableMarkup("The title of the webform to create."),
      required: TRUE,
    ),
    'description' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform Description"),
      description: new TranslatableMarkup("The description of the webform to create."),
      required: FALSE,
    ),
    'url' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform Url"),
      description: new TranslatableMarkup("The url of the webform to create."),
      required: FALSE,
    ),
  ],
)]
class CreateWebform extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'webform_id' => $webform_id,
      'title' => $title,
      'description' => $description,
      'url' => $url,
    ] = $values;

    // Validate or generate the webform ID.
    if ($webform_id) {
      // Validate the provided webform ID.
      if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $webform_id)) {
        return ExecutableResult::failure($this->t('The provided webform ID is not valid. It must start with a letter or underscore, and contain only letters, numbers, and underscores.'));
      }
    }
    else {
      // Generate a webform ID from the title.
      $webform_id = preg_replace('/[^a-zA-Z0-9_]+/', '_', strtolower($title));
      $webform_id = substr($webform_id, 0, 32);
    }

    try {
      // Create the webform.
      $webform_storage = $this->entityTypeManager->getStorage('webform');

      // Ensure the webform ID is unique.
      $existing_webform = $webform_storage->load($webform_id);
      if ($existing_webform) {
        return ExecutableResult::failure($this->t('A webform with the ID %id already exists. Please choose a different ID.', [
          '%id' => $webform_id,
        ]));
      }

      // Ensure the url is unique if provided.
      if ($url) {
        $existing_webforms = $webform_storage->loadByProperties(['url' => $url]);
        if (!empty($existing_webforms)) {
          return ExecutableResult::failure($this->t('A webform with the URL %url already exists. Please choose a different URL.', [
            '%url' => $url,
          ]));
        }
      }

      $webform = $webform_storage->create([
        'id' => $webform_id,
        'title' => $title,
        'description' => $description,
        'status' => TRUE,
      ]);
      $webform->save();
      return ExecutableResult::success($this->t('Webform %title created successfully with ID %webform_id.', [
        '%title' => $title,
        '%webform_id' => $webform_id,
      ]), ['result' => []]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('An error occurred while creating the webform: %message', ['%message' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs one of "create_webform", "edit_any_webform", or
    // "edit_own_webform" permissions.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'create webform',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
