<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the Edit Element on Webform tool.
 */
#[Tool(
  id: 'ai_webform_agent:edit_element_on_webform',
  label: new TranslatableMarkup('Edit Element of Webform'),
  description: new TranslatableMarkup('This method edits an existing element on a specified webform. Do not use this to change position of elements.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'webform_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform ID"),
      description: new TranslatableMarkup("The machine name of the webform to add elements to."),
      required: TRUE,
    ),
    'element_title' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Element Title"),
      description: new TranslatableMarkup("The title of the webform element."),
      required: TRUE,
    ),
    'element_key' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Element Key"),
      description: new TranslatableMarkup("The machine name (key) for the webform element."),
      required: TRUE,
    ),
    'element_description' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Element Description"),
      description: new TranslatableMarkup("A description for the webform element."),
      required: FALSE,
    ),
    'options' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Options"),
      description: new TranslatableMarkup("For select elements, a YAMLformatted string of options (e.g., 'option1: Option 1\noption2: Option 2')."),
      required: FALSE,
    ),
    'required' => new InputDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Required"),
      description: new TranslatableMarkup("Whether the element is required."),
      required: FALSE,
    ),
    'position' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Position"),
      description: new TranslatableMarkup("The position of this element within the form."),
      required: FALSE,
    ),
  ],
)]
class EditElementOnForm extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'webform_id' => $webform_id,
      'element_title' => $element_title,
      'element_key' => $element_key,
      'options' => $options,
      'required' => $required,
      'element_description' => $element_description,
      'position' => $position,
    ] = $values;

    // If the options are provided as a YAML string, parse them.
    if ($options) {
      $options = trim($options);
      if (str_starts_with($options, '{') || str_starts_with($options, '[')) {
        $options = json_decode($options, TRUE);
      }
      else {
        $options = Yaml::parse($options);
      }
      if (!is_array($options)) {
        return ExecutableResult::failure($this->t("Invalid options format. Please provide a valid YAML or JSON string."), []);
      }
    }

    // Load the webform entity.
    $webform_storage = $this->entityTypeManager->getStorage('webform');
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $webform_storage->load($webform_id);
    if (!$webform) {
      return ExecutableResult::failure($this->t("Webform with ID %webform_id not found.", ['%webform_id' => $webform_id]), []);
    }

    // Load existing elements.
    $elements = $webform->getElementsDecoded();
    if (isset($elements[$element_key])) {
      return ExecutableResult::failure($this->t("Element with key %elm already exists in the webform.", ['%elm' => $element_key]), []);
    }

    $existing_element = $elements[$element_key];
    if ($element_description) {
      $existing_element['#description'] = $element_description;
    }
    if ($required) {
      $existing_element['#required'] = TRUE;
    }
    else {
      unset($existing_element['#required']);
    }
    if ($element_title) {
      $existing_element['#title'] = $element_title;
    }
    if ($options) {
      $existing_element['#options'] = $options;
    }
    if ($position) {
      $existing_element['#weight'] = $position;
    }

    // Update the element.
    $elements[$element_key] = $existing_element;
    // Update the webform with the new elements.
    try {
      $webform->setElements($elements);
      $webform->save();

      return ExecutableResult::success($this->t("Element %title updated on webform %id.", [
        '%title' => $element_title,
        '%id' => $webform_id,
      ]), ['result' => []]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t("Failed to update element: %error", ['%error' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs one of "edit_any_webform", or
    // "edit_own_webform" permissions.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'edit any webform',
      'edit own webform',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
