<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the list webform elements tool.
 */
#[Tool(
  id: 'ai_webform_agent:list_webform_elements',
  label: new TranslatableMarkup('List/Search Webform Elements'),
  description: new TranslatableMarkup('Find webform by elements or list all elements available in webforms.'),
  operation: ToolOperation::Read,
  input_definitions: [
    'output_fields' => new InputDefinition(
      data_type: 'list',
      label: new TranslatableMarkup("Output Fields"),
      description: new TranslatableMarkup("The output fields to include in the response. By default, all fields are included."),
      required: FALSE,
    ),
  ],
)]
class ListWebformElements extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The webform element manager interface.
   *
   * @var \Drupal\webform\Plugin\WebformElementManagerInterface
   */
  protected WebformElementManagerInterface $webformElementManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->webformElementManager = $container->get('plugin.manager.webform.element');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    ['output_fields' => $output_fields] = $values;

    try {
      $results = [];

      // Get all webforms.
      /** @var \Drupal\webform\WebformInterface $webform_storage */
      $webform_storage = $this->entityTypeManager->getStorage('webform');
      $webforms = $webform_storage->loadMultiple();

      foreach ($webforms as $webform_id => $webform) {
        $elements = $webform->getElementsDecodedAndFlattened();

        if (empty($output_fields)) {
          // If no specific output fields are requested, include all fields.
          $results[$webform_id] = $elements;
        }
        else {
          foreach ($elements as $key => $element) {
            if (in_array($key, $output_fields)) {
              $results[$webform_id] = $elements;
            }
          }
        }
      }

      return ExecutableResult::success($this->t("The following webform elements are available:\n\n@elements", [
        '@elements' => Yaml::dump($results, 10, 2),
      ]), ['result' => $results]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('Error getting webform elements: @message', [
        '@message' => $e->getMessage(),
      ]), []);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs one of "create_webform", "edit_any_webform", or
    // "edit_own_webform" permissions.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'create webform',
      'edit any webform',
      'edit own webform',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
