<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the list webform settings tool.
 */
#[Tool(
  id: 'ai_webform_agent:list_webform_settings',
  label: new TranslatableMarkup('List Webform Settings'),
  description: new TranslatableMarkup('List all the settings available in webforms.'),
  operation: ToolOperation::Read,
)]
class ListWebformSettings extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The typed configuration manager interface.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected TypedConfigManagerInterface $configTyped;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->configTyped = $container->get('config.typed');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    try {
      $definitions = $this->configTyped->getDefinition('webform.settings');
      if (!count($definitions)) {
        return ExecutableResult::failure($this->t('No definitions where found'), []);
      }
      return ExecutableResult::success($this->t("The following webform configuration is available:\n\n@elements", [
        '@elements' => Yaml::dump($definitions, 10, 2),
      ]), ['result' => $definitions]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t('An error occurred while retrieving webform settings: @message', [
        '@message' => $e->getMessage(),
      ]), []);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs one of "create_webform", "edit_any_webform", or
    // "edit_own_webform" permissions.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'create webform',
      'edit any webform',
      'edit own webform',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
