<?php

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the list webform submissions tool.
 */
#[Tool(
  id: 'ai_webform_agent:list_webform_submissions',
  label: new TranslatableMarkup('List Webform Submissions'),
  description: new TranslatableMarkup('List all available webform elements, with an option to filter by specific output fields.'),
  operation: ToolOperation::Read,
  input_definitions: [
    'webform_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform ID"),
      description: new TranslatableMarkup("The machine name of the webform to list submissions from."),
      required: TRUE,
    ),
    'page_size' => new InputDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup("Page Size"),
      description: new TranslatableMarkup("The number of submissions to return per page."),
      required: FALSE,
      default_value: 10,
    ),
    'page_number' => new InputDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup("Page Number"),
      description: new TranslatableMarkup("The page number to return."),
      required: FALSE,
      default_value: 1,
    ),
  ],
  output_definitions: [],
)]
class ListWebformSubmissions extends ToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->dateFormatter = $container->get('date.formatter');
    return $instance;
  }

  /**
   * Execute function.
   *
   * @param array $values
   *   The input values.
   *
   * @return \Drupal\tool\ExecutableResult
   *   The submissions information.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'webform_id' => $webform_id,
      'page_size' => $page_size,
      'page_number' => $page_number,
    ] = $values;

    // Load the webform entity.
    $webform_storage = $this->entityTypeManager->getStorage('webform');
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $webform_storage->load($webform_id);
    if (!$webform) {
      throw new \Exception("Webform with ID '$webform_id' not found.");
    }

    // Check permissions.
    if (
      !$this->currentUser->hasPermission('edit any webform') &&
      (!$this->currentUser->hasPermission('edit own webform') || $webform->getOwnerId() != $this->currentUser->id())
    ) {
      return ExecutableResult::failure($this->t("You do not have permission to access this tool."), []);
    }
    // Load submissions.
    $submission_storage = $this->entityTypeManager->getStorage('webform_submission');
    $query = $submission_storage->getQuery()
      ->condition('webform_id', $webform_id)
      ->range(($page_number - 1) * $page_size, $page_size)
      ->accessCheck(TRUE)
      ->sort('created', 'DESC');
    $submission_ids = $query->execute();
    // Get a total count as well.
    $total_count = $submission_storage->getQuery()
      ->condition('webform_id', $webform_id)
      ->accessCheck(TRUE)
      ->count()
      ->execute();
    /** @var \Drupal\webform\WebformSubmissionInterface[] $submissions */
    $submissions = $submission_storage->loadMultiple($submission_ids);
    if (count($submissions) == 0) {
      return ExecutableResult::failure($this->t("The webform %webform_id has no submissions.", ['%webform_id' => $webform_id]), []);
    }
    $result = [];
    foreach ($submissions as $submission) {
      $id = $submission->id();
      $data = $submission->getData();
      $created = $this->dateFormatter->format($submission->getCreatedTime(), 'short');
      $result[] = [
        'id' => $id,
        'created' => $created,
        'data' => $data,
      ];
    }
    $output = [
      'total_count' => $total_count,
      'page_number' => $page_number,
      'page_size' => $page_size,
      'submissions' => $result,
    ];
    $yaml_dump = Yaml::dump($output, 10, 2);
    return ExecutableResult::success($this->t('Success. %yaml', ['%yaml' => $yaml_dump]), [$output]);
  }

  /**
   * Checks for access.
   *
   * @param array $values
   *   The input values.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user account.
   * @param bool $return_as_object
   *   If to return as object or not.
   *
   * @aram $return_as_object
   *  AccessResult.
   *
   * @return bool|\Drupal\Core\Access\AccessResultInterface
   *   The return value.
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // Basic permission check - more specific access is done per entity.
    $account = $this->currentUser;
    // Allow if user has general access to view content.
    $access_result = AccessResult::allowedIfHasPermission($account, 'create webform');
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

}
