<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the Set Webform Settings tool.
 */
#[Tool(
  id: 'ai_webform_agent:set_settings_to_form',
  label: new TranslatableMarkup('Set Settings to Webform'),
  description: new TranslatableMarkup('This adds a new element to an existing webform with specified settings.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'webform_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform ID"),
      description: new TranslatableMarkup("The machine name of the webform to add settings to."),
      required: TRUE,
    ),
    'setting_key' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Setting Key"),
      description: new TranslatableMarkup("The key of the setting to set."),
      required: TRUE,
    ),
    'setting_value' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Setting Value"),
      description: new TranslatableMarkup("The value of the setting to set. For complex settings, provide a YAML string."),
      required: TRUE,
    ),
  ],
)]
class SetWebformSettings extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'webform_id' => $webform_id,
      'setting_key' => $setting_key,
      'setting_value' => $setting_value,
    ] = $values;

    // Load the webform entity.
    $webform_storage = $this->entityTypeManager->getStorage('webform');
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $webform_storage->load($webform_id);
    if (!$webform) {
      return ExecutableResult::failure($this->t("Webform with ID @webform_id not found.", ['@webform_id' => $webform_id]));
    }

    // Parse the settings value if it's in YAML format.
    $parsed_value = Yaml::parse($setting_value);
    if ($parsed_value === NULL) {
      // If parsing fails, treat it as a simple string.
      $parsed_value = $setting_value;
    }

    // Update the webform with the new setting.
    try {
      $webform->setSetting($setting_key, $parsed_value);
      $webform->save();

      return ExecutableResult::success($this->t("Successfully set settings @settings_key on webform @webform_id.", [
        '@settings_key' => $setting_key,
        '@webform_id' => $webform_id,
      ]), ['result' => []]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t("Failed to set setting: %error", ['%error' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // The user needs "administer webform" permission.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'administer webform',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
