<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the Submit Webform Submission tool.
 */
#[Tool(
  id: 'ai_webform_agent:submit_webform_submission',
  label: new TranslatableMarkup('Submit Webform Submission'),
  description: new TranslatableMarkup('Submit a new submission to a specified webform with given element values.'),
  operation: ToolOperation::Write,
  input_definitions: [
    'webform_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform ID"),
      description: new TranslatableMarkup("The machine name of the webform to add elements to."),
      required: TRUE,
    ),
    'data' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Submission Data"),
      description: new TranslatableMarkup("A YAML or JSON formatted string representing the submission data, where keys are element keys and values are the corresponding values to submit."),
      required: TRUE,
    ),
  ],
)]
class SubmitWebformSubmission extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'webform_id' => $webform_id,
      'data' => $data,
    ] = $values;

    // Load the webform entity.
    $webform_storage = $this->entityTypeManager->getStorage('webform');
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $webform_storage->load($webform_id);
    if (!$webform) {
      return ExecutableResult::failure($this->t("Webform with ID @webform_id not found.", ['@webform_id' => $webform_id]));
    }

    // @todo check if the webform has the fields supplied in the data.
    // Update the webform with the new elements.
    try {
      $submissions = $this->entityTypeManager->getStorage('webform_submission');
      $submission = $submissions->create([
        'webform_id' => $webform_id,
        'data' => is_string($data) ? (Yaml::parse($data) ?: json_decode($data, TRUE)) : $data,
        'in_draft' => FALSE,
      ]);
      $submission->save();
      return ExecutableResult::success($this->t("Submission created with ID: @id.", [
        '@id' => $submission->id(),
      ]), ['result' => []]);
    }
    catch (\Exception $e) {
      return ExecutableResult::failure($this->t("Failed to create submission: %error", ['%error' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    // @todo add access check.
    $access = AccessResult::allowedIfHasPermissions($account, [
      'access content',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
