<?php

declare(strict_types=1);

namespace Drupal\ai_webform_agent\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the list webform settings tool.
 */
#[Tool(
  id: 'ai_webform_agent:view_settings_on_form',
  label: new TranslatableMarkup('View Settings on Webform'),
  description: new TranslatableMarkup('This retrieves the settings of an existing webform.'),
  operation: ToolOperation::Read,
  input_definitions: [
    'webform_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Webform ID"),
      description: new TranslatableMarkup("The machine name of the webform to add elements to."),
      required: TRUE,
    ),
  ],
)]
class ViewSettingsOnForm extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'webform_id' => $webform_id,
    ] = $values;

    // Load the webform entity.
    $webform_storage = $this->entityTypeManager->getStorage('webform');
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $webform_storage->load($webform_id);
    if (!$webform) {
      return ExecutableResult::failure($this->t("Webform with ID @webform_id not found.", ['@webform_id' => $webform_id]));
    }

    $elements = $webform->getSettings();
    return ExecutableResult::success($this->t("Successfully retrieved the webform settings: @elements.", [
      '@elements' => Yaml::dump($elements, 10, 2),
    ]), ['result' => $elements]);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account ??= $this->currentUser;

    $access = AccessResult::allowedIfHasPermissions($account, [
      'access webform overview',
    ], 'OR');
    return $return_as_object ? $access : $access->isAllowed();
  }

}
