<?php

declare(strict_types=1);

namespace Drupal\ai_webform_guard\Hook;

use Drupal\ai_webform_guard\Service\ProviderHelperInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;

/**
 * Hook implementations for ai_webform_guard.
 */
final class AiWebformGuardFormHooks {
  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Constructs a new AiWebformGuardFormHooks object.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected ProviderHelperInterface $providerHelper,
    protected LoggerChannelFactoryInterface $loggerFactory,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * Implements hook_webform_submission_form_alter().
   */
  #[Hook('webform_submission_form_alter')]
  public function alterWebformSubmissionForm(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Add this object as the validator - DependencySerializationTrait
    // will handle serialization.
    // Add a checkbox field to the form.
    $config = $this->configFactory->get('ai_webform_guard.settings');
    if ($config->get('human_iteration')) {
      $form['spam_confirm'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('I confirm that this submission is not spam.'),
        '#access' => FALSE,
        '#required' => FALSE,
        '#default_value' => $form_state->getUserInput()['spam_confirm'] ?? FALSE,
        '#weight' => -100,
      ];
    }

    $form['#validate'][] = [$this, 'validateSubmission'];
  }

  /**
   * Custom validation handler for webform submissions.
   */
  public function validateSubmission(array &$form, FormStateInterface $form_state): void {
    // Get the webform submission.
    $submission = $form_state->getFormObject()?->getEntity();

    // Skip validation if this is not a webform submission.
    if (!$submission instanceof WebformSubmissionInterface) {
      return;
    }

    // Check for any errors in the form state.
    if ($form_state->hasAnyErrors()) {
      return;
    }

    $user_input = $form_state->getUserInput();
    $spam_confirm = isset($user_input['spam_confirm']) && $user_input['spam_confirm'];

    // If spam_confirm is checked, skip spam classification validation.
    if ($spam_confirm) {
      return;
    }

    $webform_id = $submission->getWebform()->id();
    $config = $this->configFactory->get('ai_webform_guard.settings');
    $fields_config = $this->configFactory->get('ai_webform_guard.fields_settings')->get('settings_fields') ?? [];
    $excluded_fields = $fields_config[$webform_id]['excluded_fields'] ?? [];

    // Get the prompt configuration.
    $custom_prompt = $fields_config[$webform_id]['custom_prompt'] ?? NULL;

    $provider_data = $this->providerHelper->getSetProvider();
    if (!$provider_data) {
      $form_state->setErrorByName('', $this->t('No AI provider configured.'));
      return;
    }

    $provider = $provider_data['provider_id'];
    $model = $provider_data['model_id'];

    $prompt = $custom_prompt ?: $config->get('prompt');
    $prompt .= "\n" . $this->t('Based on your analysis, respond with one of the following classifications:') . "\n";
    $prompt .= "Classification: Spam\nClassification: Not Spam\n";

    $data = $submission->getData();
    $submission_text = '';
    foreach ($data as $key => $value) {
      if (!empty($excluded_fields[$key])) {
        continue;
      }
      $submission_text .= ucfirst($key) . ': ' . (is_array($value) ? implode(', ', $value) : $value) . "\n";
    }

    // Limit text length based on max_words config.
    $max_words = (int) ($config->get('max_words') ?? 500);

    if ($max_words > 0) {
      // Normalize whitespace and split into words.
      $words = preg_split('/\s+/', trim($submission_text));
      if (count($words) > $max_words) {
        $submission_text = implode(' ', array_slice($words, 0, $max_words));
        $submission_text .= "\n\n[Truncated to first {$max_words} words for AI analysis]";
      }
    }

    $input = new ChatInput([
      new ChatMessage('user', $prompt . "\n\n" . $submission_text),
    ]);

    $output = $provider->chat($input, $model);

    $response_text = $output->getNormalized()->getText();

    if (preg_match('/Classification:\s*(Spam|Not Spam)/i', $response_text, $matches)) {
      $classification = strtolower(trim($matches[1]));
      if ($classification === 'spam') {
        $message = $config->get('error_message') ?: $this->t('This submission has been identified as spam.');
        $form_state->setErrorByName('', $message);

        // Add a custom error message to the form state.
        if ($config->get('human_iteration')) {
          $form['spam_confirm']['#access'] = TRUE;
          $form['spam_confirm']['#required'] = TRUE;
        }

        if ($config->get('log_spam_attempts')) {
          $this->loggerFactory->get('ai_webform_guard')->notice(
            'Spam attempt blocked on webform @id. Full AI response: @response',
            ['@id' => $webform_id, '@response' => $response_text]
          );
        }
      }
    }

  }

}
