<?php

declare(strict_types=1);

namespace Drupal\ai_form_guard\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a settings form for AI Form Guard custom forms.
 */
final class AIFormGuardFieldsSettingsForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_form_guard.fields_settings';

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new AIFormGuardFieldsSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed config manager service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    parent::__construct($config_factory, $typed_config);
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('messenger'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_form_guard_fields_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);
    $settings_fields_config = $config->get('settings_fields') ?? [];

    // Get custom form IDs from ai_form_guard.settings.
    $custom_form_ids_raw = $this->config('ai_form_guard.settings')->get('custom_form_ids') ?? '';
    $custom_form_ids = array_filter(array_map('trim', explode("\n", $custom_form_ids_raw)));

    if (empty($custom_form_ids)) {
      $form['empty_message'] = [
        '#markup' => '<p>' . $this->t('No custom form IDs configured. Please configure custom form IDs in the <a href=":settings_url">AI Webform Guard settings</a> first.',
            [
              ':settings_url' => Url::fromRoute('ai_webform_guard.settings')->toString(),
            ]) . '</p>',
      ];
      return $form;
    }

    $form['description'] = [
      '#markup' => '<h4>' . $this->t('Choose the custom form fields to exclude from AI spam detection, and optionally define a custom AI prompt for each form.') . '</h4>',
    ];

    $form['settings_fields'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
    ];

    foreach ($custom_form_ids as $form_id) {
      $form['settings_fields'][$form_id] = [
        '#type' => 'details',
        '#title' => $this->t('Form: @form_id', ['@form_id' => $form_id]),
        '#open' => FALSE,
      ];

      $form['settings_fields'][$form_id]['info'] = [
        '#markup' => '<p><em>' . $this->t('Enter the field names to exclude from spam detection. Field names will be validated at runtime.') . '</em></p>',
      ];

      // Textarea for excluded field names.
      $excluded_fields = $settings_fields_config[$form_id]['excluded_fields'] ?? [];
      $default_value = !empty($excluded_fields) ? implode("\n", array_keys($excluded_fields)) : '';

      $form['settings_fields'][$form_id]['excluded_fields'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Excluded fields'),
        '#description' => $this->t('Enter one field name per line. These fields will be ignored by AI spam detection. Example: form_build_id, form_token, op'),
        '#default_value' => $default_value,
        '#rows' => 6,
      ];

      $form['settings_fields'][$form_id]['custom_prompt'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Custom prompt'),
        '#default_value' => $settings_fields_config[$form_id]['custom_prompt'] ?? '',
        '#description' => $this->t('Custom prompt used for generating the spam detection for this form. Leave empty to use the default prompt.'),
        '#rows' => 5,
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $settings_fields = $form_state->getValue('settings_fields') ?? [];

    // Convert textarea input to array format.
    foreach ($settings_fields as &$settings) {
      if (!empty($settings['excluded_fields'])) {
        $lines = array_filter(array_map('trim', explode("\n", $settings['excluded_fields'])));
        $settings['excluded_fields'] = array_fill_keys($lines, TRUE);
      }
      else {
        $settings['excluded_fields'] = [];
      }
    }

    $this->config(static::CONFIG_NAME)
      ->set('settings_fields', $settings_fields)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
