<?php

declare(strict_types=1);

namespace Drupal\ai_form_guard\Hook;

use Drupal\ai_webform_guard\Service\ProviderHelperInterface;
use Drupal\ai_webform_guard\Service\SpamDetectionServiceInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Hook implementations for ai_form_guard.
 */
final class AiFormGuardFormHooks {
  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Constructs a new AiFormGuardFormHooks object.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected ProviderHelperInterface $providerHelper,
    protected LoggerChannelFactoryInterface $loggerFactory,
    TranslationInterface $stringTranslation,
    protected SpamDetectionServiceInterface $spamDetectionService,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * Implements hook_form_FORM_ID_alter() for ai_webform_guard_settings.
   */
  #[Hook('form_ai_webform_guard_settings_alter')]
  public function formAiWebformGuardSettingsAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Add custom form IDs field to the existing settings form.
    $form['custom_form_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Form IDs'),
      '#description' => $this->t('Enter the form IDs of custom forms to protect with AI spam detection. One form ID per line. Example: contact_form, user_register_form'),
      '#default_value' => $this->configFactory->get('ai_form_guard.settings')->get('custom_form_ids') ?? '',
      '#rows' => 5,
      '#weight' => 10,
    ];

    // Add custom submit handler to save our custom field.
    $form['#submit'][] = [$this, 'settingsSubmit'];
  }

  /**
   * Custom submit handler for ai_webform_guard_settings form.
   */
  public function settingsSubmit(array &$form, FormStateInterface $form_state): void {
    $this->configFactory->getEditable('ai_form_guard.settings')
      ->set('custom_form_ids', $form_state->getValue('custom_form_ids'))
      ->save();
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Get custom form IDs from configuration.
    $custom_form_ids_raw = $this->configFactory->get('ai_form_guard.settings')->get('custom_form_ids') ?? '';
    $custom_form_ids = array_filter(array_map('trim', explode("\n", $custom_form_ids_raw)));

    // Check if this form should be protected.
    if (!in_array($form_id, $custom_form_ids, TRUE)) {
      return;
    }

    // Add human iteration checkbox if enabled.
    $config = $this->configFactory->get('ai_webform_guard.settings');
    if ($config->get('human_iteration')) {
      $form['spam_confirm'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('I confirm that this submission is not spam.'),
        '#access' => FALSE,
        '#required' => FALSE,
        '#default_value' => $form_state->getUserInput()['spam_confirm'] ?? FALSE,
        '#weight' => -100,
      ];
    }

    // Add validation handler.
    $form['#validate'][] = [$this, 'validateCustomFormSubmission'];
  }

  /**
   * Custom validation handler for custom form submissions.
   */
  public function validateCustomFormSubmission(array &$form, FormStateInterface $form_state): void {
    $form_id = $form['#form_id'] ?? $form_state->getFormObject()?->getFormId();

    if (!$form_id) {
      return;
    }

    // Check for any errors in the form state.
    if ($form_state->hasAnyErrors()) {
      return;
    }

    // Check spam_confirm checkbox.
    $user_input = $form_state->getUserInput();
    $spam_confirm = isset($user_input['spam_confirm']) && $user_input['spam_confirm'];

    if ($spam_confirm) {
      return;
    }

    // Get configurations.
    $fields_config = $this->configFactory->get('ai_form_guard.fields_settings')->get('settings_fields') ?? [];
    $excluded_fields = $fields_config[$form_id]['excluded_fields'] ?? [];
    $custom_prompt = $fields_config[$form_id]['custom_prompt'] ?? NULL;

    // Get AI provider.
    $provider_data = $this->providerHelper->getSetProvider();
    if (!$provider_data) {
      $form_state->setErrorByName('', $this->t('No AI provider configured.'));
      return;
    }

    // Get max_words from config.
    $config = $this->configFactory->get('ai_webform_guard.settings');
    $max_words = (int) ($config->get('max_words') ?? 500);

    // Use SpamDetectionService to detect spam.
    try {
      $values = $form_state->getValues();

      $result = $this->spamDetectionService->detectSpam($values, [
        'excluded_fields' => $excluded_fields,
        'custom_prompt' => $custom_prompt,
        'max_words' => $max_words,
        'form_id' => $form_id,
      ]);

      $is_spam = $result['is_spam'];
      $response_text = $result['response_text'];

      if ($is_spam) {
        $this->spamDetectionService->handleSpamDetection($form, $form_state, $form_id, $is_spam, $response_text);
      }
    }
    catch (\Exception $e) {
      // Error is already logged in SpamDetectionService.
      $form_state->setErrorByName('', $this->t('An error occurred during spam detection.'));
    }
  }

}
