<?php

namespace Drupal\ai_webform_guard\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\webform\Entity\Webform;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a settings form for AI Webform Guard.
 */
final class AIWebformGuardFieldsSettingsForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_webform_guard.fields_settings';

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new AIWebformGuardFieldsSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed config manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config, ModuleHandlerInterface $module_handler, MessengerInterface $messenger) {
    parent::__construct($config_factory, $typed_config);
    $this->moduleHandler = $module_handler;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('module_handler'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_webform_guard_fields_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);
    $settings_fields_config = $config->get('settings_fields') ?? [];

    $form['description'] = [
      '#markup' => '<h4>' . $this->t("Choose the webform fields to exclude from AI spam detection, and optionally define a custom AI prompt for each webform.") . '</h4>',
    ];

    $form['settings_fields'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
    ];

    $webforms = Webform::loadMultiple();

    foreach ($webforms as $webform_id => $webform) {
      $form['settings_fields'][$webform_id] = [
        '#type' => 'details',
        '#title' => $webform->label(),
        '#open' => FALSE,
      ];

      $elements = $webform->getElementsInitializedFlattenedAndHasValue();

      foreach ($elements as $key => $element) {
        $form['settings_fields'][$webform_id]['excluded_fields'][$key] = [
          '#type' => 'checkbox',
          '#title' => $element['#title'] ?? $key,
          '#default_value' => $settings_fields_config[$webform_id]['excluded_fields'][$key] ?? FALSE,
        ];
      }

      $form['settings_fields'][$webform_id]['custom_prompt'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Custom prompt'),
        '#default_value' => $settings_fields_config[$webform_id]['custom_prompt'] ?? '',
        '#description' => $this->t('Custom prompt used for generating the spam detection.'),
      ];

      // Check if webform_email_confirmation_link module is available.
      $confirmation_module_exists = $this->moduleHandler->moduleExists('webform_email_confirmation_link');

      $form['settings_fields'][$webform_id]['enable_spam_confirmation_flow'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable spam confirmation flow'),
        '#default_value' => $settings_fields_config[$webform_id]['enable_spam_confirmation_flow'] ?? FALSE,
        '#disabled' => !$confirmation_module_exists,
        '#description' => $this->t('If enabled, the email confirmation handler will be automatically enabled or disabled based on spam detection. When spam is detected, the handler is enabled to require email verification. When no spam is detected, the handler remains disabled for normal form processing. <strong>Note:</strong> This feature requires the <a href="@module_link" target="_blank">Webform Email Confirmation Link</a> module to be installed. See the <a href="@readme_link" target="_blank">README</a> for detailed configuration instructions and examples.', [
          '@module_link' => 'https://www.drupal.org/project/webform_email_confirmation_link',
          '@readme_link' => 'https://git.drupalcode.org/project/ai_webform_guard/-/blob/1.1.x/README.md#spam-confirmation-flow-per-webform',
        ]),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $excluded_fields = $form_state->getValue('settings_fields') ?? [];

    $this->config(static::CONFIG_NAME)
      ->set('settings_fields', $excluded_fields)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
